package storage

import (
	"fmt"

	"code.justin.tv/esports-exp/centaur/internal/config"

	"github.com/google/uuid"
	"github.com/jinzhu/gorm"
	_ "github.com/jinzhu/gorm/dialects/postgres"
)

type Gorm struct {
	db *gorm.DB
}

func New(cfg *config.Config) (*Gorm, error) {
	db, err := gorm.Open("postgres", fmt.Sprintf("host=%s user=%s password=%s sslmode=disable ",
		cfg.DBHost, cfg.DBUser, cfg.DBPassword))
	if err != nil {
		return nil, err
	}

	db.AutoMigrate(&User{})
	db.AutoMigrate(&Clip{})
	db.AutoMigrate(&Reaction{})
	db.AutoMigrate(&Watched{})

	db.Exec("ALTER TABLE reactions DROP CONSTRAINT IF EXISTS emote_id_clip_id;")
	db.Exec("ALTER TABLE reactions ADD CONSTRAINT emote_id_clip_id_user_id UNIQUE (emote_id, clip_id, user_id);")

	// db.Model(&Reaction{}).AddUniqueIndex("clip_id_emote_id", "clip_id", "emote_id")

	return &Gorm{
		db,
	}, nil
}

func (g *Gorm) GetUserByDeviceID(deviceID string) (*User, error) {
	var user User
	err := g.db.Where(&User{
		DeviceID: deviceID,
	}).
		First(&user).
		Error

	if gorm.IsRecordNotFoundError(err) {
		user.DeviceID = deviceID
		err = g.db.Create(&user).
			Error
	}

	return &user, err
}

func (g *Gorm) GetClipsFromClipIDS(clipIDS []string) ([]*Clip, error) {
	clips := make([]*Clip, len(clipIDS), len(clipIDS))
	err := g.db.Where("clips.id in (?) AND clips.video_url IS NOT NULL AND clips.thumbnail_url IS NOT NULL", clipIDS).
		Find(&clips).
		Error

	if err != nil {
		return nil, err
	}

	return clips, nil
}

func (g *Gorm) SaveClip(clip *Clip) error {
	err := g.db.
		Model(&Clip{}).
		Where(&Clip{
			Slug: clip.Slug,
		}).
		Find(&Clip{}).
		Error
	if gorm.IsRecordNotFoundError(err) {
		return g.db.Create(clip).Error
	} else if err != nil {
		return err
	}

	return g.db.Model(clip).Where("clips.slug = ?", clip.Slug).Update(clip).First(clip).Error
}

func (g *Gorm) GetClipByClipID(clipID string) (*Clip, error) {
	var clip Clip

	id, err := uuid.Parse(clipID)
	if err != nil {
		return nil, err
	}

	err = g.db.Model(&Clip{}).
		Where(&Clip{
			GormBase: GormBase{
				ID: id,
			},
		}).
		First(&clip).
		Error

	return &clip, err
}

func (g *Gorm) MarkClipWatchedByUser(clip *Clip, user *User, duration int64) error {
	return g.db.Model(&Watched{}).
		Create(&Watched{
			Duration: duration,
			Clip:     clip,
			User:     user,
		}).
		Error
}

func (g *Gorm) GetUserWatched(user *User, clip *Clip) (*Watched, error) {
	var watched Watched
	err := g.db.Model(&Watched{}).
		Where(&Watched{
			UserID: user.ID,
			ClipID: clip.ID,
		}).
		First(&watched).
		Error

	if gorm.IsRecordNotFoundError(err) {
		return nil, nil
	}

	if err != nil {
		return nil, err
	}

	return &watched, nil
}
