package storage

import "github.com/jinzhu/gorm"

type User struct {
	GormBase

	DeviceID string `gorm:"index:device_id"`
}

func (user *User) CountWatchedClips(g *Gorm) (int64, error) {
	var out int64

	err := g.db.Model(&Clip{}).
		Joins("LEFT JOIN watcheds ON watcheds.clip_id = clips.id AND watcheds.user_id = ?", user.ID.String()).
		Where("watcheds.id IS NOT NULL").
		Count(&out).
		Error

	return out, err
}

// randomly select N from top 100 clips of each category
// SELECT *, ROW_NUMBER() over (partition by game order by views desc) as rank
// FROM clips where rank < 100
// LEFT JOIN watcheds ON watcheds.clip_id = clips.id AND watcheds.user_id = ?
// WHERE watcheds.id IS NULL
// ORDER BY random()
// LIMIT ?

func (user *User) GetRandomNewClips(g *Gorm, limit int64) ([]*Clip, error) {
	var clips []*Clip
	err := g.db.Set("gorm:auto_preload", true).
		Joins("LEFT JOIN watcheds ON watcheds.clip_id = clips.id AND watcheds.user_id = ?", user.ID.String()).
		Where("watcheds.id IS NULL and clips.views > 10000 AND clips.video_url IS NOT NULL AND clips.thumbnail_url IS NOT NULL").
		Order(gorm.Expr("random()")).
		Limit(limit).
		Find(&clips).
		Error

	return clips, err
}

func (user *User) GetRandomWeightedClips(g *Gorm, limit int64) ([]*Clip, error) {
	rows, err := g.db.Raw(`
	WITH ranked_clips AS
	(SELECT *,
			ROW_NUMBER() over (partition BY game
							 ORDER BY views DESC) AS rank
	 FROM clips
	 LEFT JOIN watcheds
	   ON watcheds.clip_id = clips.id
	   AND watcheds.user_id = ?
	 WHERE watcheds.id IS NULL and game is not null), weighted_clips AS
	(SELECT rc.*,
			CASE
				WHEN rank < 100 AND views > 10000 THEN 7
				WHEN rank < 20  THEN 7
				WHEN rank < 100 THEN 4
				WHEN rank < 500 AND views > 1000 THEN 3
				WHEN rank < 200 THEN 2
				ELSE 1
			END AS weight
	 FROM ranked_clips rc)
  SELECT *
  FROM weighted_clips wc
  ORDER BY -LOG(1.0 - RANDOM()) / weight 
  LIMIT ?;
	`, user.ID.String(), limit).Rows()
	if err != nil {
		return nil, err
	}
	defer rows.Close()

	var out []*Clip
	for rows.Next() {
		var clip Clip
		g.db.ScanRows(rows, &clip)
		out = append(out, &clip)
	}
	return out, nil
}
