package main

import (
	"context"
	"fmt"
	"net/http"
	"os"

	"code.justin.tv/esports-exp/centaur/internal/config"
	"code.justin.tv/esports-exp/centaur/internal/storage"
	"github.com/cheggaaa/pb/v3"
	"github.com/machinebox/graphql"
	"github.com/nicklaw5/kraken"
)

var (
	pages        = 10
	pageSize     = 100
	defaultScore = 0.5
	gqlClient    = graphql.NewClient("https://gql.twitch.tv/gql")

	games = []string{
		"Just Chatting",
		"League of Legends",
		"Fortnite",
		"VALORANT",
		"Call of Duty: Modern Warfare",
		"Grand Theft Auto V",
		"Counter-Strike: Global Offensive",
		"Hearthstone",
		"Apex Legends",
		"Dota 2",
		"Sea of Thieves",
		"Super Mario 64",
		"Teamfight Tactics",
		"Overwatch",
		"World of Warcraft",
		"NBA 2K20",
		"Terraria",
		"Music & Performing Arts",
		"Dead by Daylight",
		"Dragon Ball FighterZ",
		"Minecraft",
		"Escape From Tarkov",
		"Monster Hunter World",
		"Animal Crossing: New Horizons",
		"Super Mario Maker 2",
		"Tom Clancy's Rainbow Six: Siege",
		"Old School RuneScape",
		"FIFA 20",
		"Getting Over It",
	}
)

func main() {
	oauth := os.Getenv("GQL_OAUTH")
	if oauth == "" {
		panic("Must export GQL_OAUTH")
	}
	cfg := config.New()
	gorm, err := storage.New(cfg)
	if err != nil {
		panic(fmt.Errorf("failed to setup gorm: %s", err))
	}

	client, err := kraken.NewClient(os.Getenv("KRAKEN_CLIENT_ID"), http.DefaultClient)
	if err != nil {
		panic(err)
	}

	clips := []kraken.Clip{}

	for _, game := range games {
		var (
			cursor     string
			reachedEnd bool
		)

		gclips := []kraken.Clip{}

		for i := 0; i < pages; i++ {
			if reachedEnd {
				fmt.Println("Early stop: ", game, i)
				break
			}
			params := &kraken.TopClipsParams{
				Limit:    pageSize,
				Cursor:   cursor,
				Game:     game,
				Language: "en,es,pt,fr",
				Period:   "month",
			}
			resp, err := client.GetTopClips(params)
			if err != nil {
				panic(err)
			}

			gclips = append(gclips, resp.Data.Clips...)
			cursor = resp.Data.Cursor
			if cursor == "" {
				reachedEnd = true
			}
		}

		clips = append(clips, gclips...)
	}

	bar := pb.StartNew(len(clips))

	for _, clip := range clips {
		sClip := &storage.Clip{
			Type:            storage.TwitchClipType,
			Slug:            clip.Slug,
			Game:            clip.Game,
			Curator:         clip.Curator.Name,
			Broadcaster:     clip.Broadcaster.Name,
			BroadcasterLogo: clip.Broadcaster.Logo,
			Title:           clip.Title,
			Views:           int64(clip.Views),
			Duration:        int64(clip.Duration),
		}

		err = addClipDetails(sClip)
		if err != nil {
			fmt.Println("Add details err", err)
		}

		err = gorm.AddReactions(sClip.Reactions)
		if err != nil {
			panic(err)
		}

		err = gorm.SaveClip(sClip)
		if err != nil {
			panic(err)
		}
		// var score float64

		// reactionDensity := float64(reactCount) / clip.Duration
		// fmt.Println(reactionDensity)

		// if reactionDensity > 20 {
		// 	score = 0.5
		// } else if reactionDensity > 10 {
		// 	score = 0.4
		// } else if reactionDensity > 1 {
		// 	score = 0.3
		// } else {
		// 	score = 0.2
		// }

		// err = rr.Rate(sClip.ID.String(), clip.Game+clip.Language, score)
		// if err != nil {
		// 	panic(err)
		// }
		bar.Increment()
	}
	bar.Finish()

	fmt.Printf("Saved %d clips :) \n", len(clips))
}

func addClipDetails(clip *storage.Clip) error {
	req := graphql.NewRequest(`
		query ClipEmotes($slug: ID!) {
			clip(slug: $slug) {
				id
				videoQualities {
					sourceURL
					quality
				}
				thumbnailURL(width:480,height:272)
				video {
					comments{
						edges {
							node {
								message {
									fragments {
										emote {
											emoteID
										}
									}
								}
							}
						}
					}
				}
			}
		}
	`)
	req.Var("slug", clip.Slug)
	req.Header.Set("Cache-Control", "no-cache")
	oauth := os.Getenv("GQL_OAUTH")
	if oauth == "" {
		panic("Must export GQL_OAUTH")
	}
	req.Header.Set("Client-Id", "kimne78kx3ncx6brgo4mv6wki5h1ko") //twilight :)
	req.Header.Set("Authorization", oauth)
	ctx := context.Background()
	var resp struct {
		Clip struct {
			ThumbnailURL   string `json:"thumbnailURL"`
			VideoQualities []struct {
				SourceURL string `json:"sourceURL"`
				Quality   string `json:"quality"`
			} `json:"videoQualities"`
			Video struct {
				Comments struct {
					Edges []struct {
						Node struct {
							Message struct {
								Fragments []struct {
									Emote struct {
										EmoteID string `json:"emoteID"`
									} `json:"emote"`
								} `json:"fragments"`
							} `json:"message"`
						} `json:"node"`
					} `json:"edges"`
				} `json:"comments"`
			} `json:"video"`
		} `json:"clip"`
	}

	if err := gqlClient.Run(ctx, req, &resp); err != nil {
		panic(err)
	}

	var cnt int64
	reactions := map[string]int64{}

	for _, edge := range resp.Clip.Video.Comments.Edges {
		for _, fragment := range edge.Node.Message.Fragments {
			emoteID := fragment.Emote.EmoteID
			if emoteID == "" {
				continue
			}

			if _, ok := reactions[emoteID]; ok {
				reactions[emoteID]++
			} else {
				reactions[emoteID] = 1
			}
			cnt++
		}
	}

	out := []*storage.Reaction{}

	for emoteID, amount := range reactions {
		out = append(out, &storage.Reaction{
			EmoteID: emoteID,
			Amount:  amount,
			ClipID:  clip.ID,
		})
	}

	for _, quality := range resp.Clip.VideoQualities {
		if quality.Quality == "720" {
			clip.VideoURL = quality.SourceURL
		}
	}

	clip.ThumbnailURL = resp.Clip.ThumbnailURL

	return nil
}
