import React, { useState, useEffect } from "react";
import createPersistedState from "use-persisted-state";
import Chance from "chance";

import "./App.scss";

import { AddEmote, GetRecommendation, ReportTimeWatched } from "./API";

const useUser = createPersistedState("user");

function App() {
  const [clipIdx, setClipIdx] = useState(0);
  const [startTime, setStartTime] = useState(new Date().getTime());
  const [clips, setClips] = useState<Array<any>>([]);
  const [user, setUser] = useUser(new Chance().animal());
  const [userInput, setUserInput] = useState(user);

  const [emoteCount, setEmoteCount] = useState(0);

  useEffect(() => {
    setUserInput(user);
    setClipIdx(0);
    loadClips(true);
  }, [user]);

  const loadClips = async (reset = false) => {
    const resp = await GetRecommendation(user, 5);
    let newClips = [...resp.clips];
    if (!reset) {
      newClips = [...clips, ...newClips];
    }
    setClips(newClips);
  };

  const prevClip = () => {
    setStartTime(new Date().getTime());
    setClipIdx(clipIdx - 1);
  };

  const nextClip = () => {
    if (clips.length === clipIdx + 2) {
      loadClips();
    }

    if (!clips[clipIdx].reported) {
      clips[clipIdx].reported = true;
      const seconds = Math.round((new Date().getTime() - startTime) / 1000);
      ReportTimeWatched(user, clips[clipIdx].id, seconds);
    }

    setStartTime(new Date().getTime());
    setClipIdx(clipIdx + 1);
  };

  useEffect(() => {
    if (clips[clipIdx]) {
      let ec = 0;
      clips[clipIdx].reactions?.forEach((reac: any) => {
        ec += parseInt(reac.amount);
      });
      setEmoteCount(ec);
    }
  }, [clips, clipIdx]);

  if (!clips[clipIdx]) {
    return <div>Loading...</div>;
  }

  const addEmote = async () => {
    await AddEmote(user, clips[clipIdx].id, "0");
    setEmoteCount(emoteCount + 1);
  };

  return (
    <div className="App">
      <header className="App-header">
        <div className="Title">WebClop</div>
        <div className="User">
          <form onClick={() => setUser(userInput)}>
            <input
              type="text"
              value={userInput}
              onChange={(e) => setUserInput(e.target.value)}
            />
            <input type="submit" value="Set User" />
          </form>
        </div>
      </header>
      <div className="Content">
        <iframe
          src={`https://clips.twitch.tv/embed?clip=${clips[clipIdx].slug}`}
          width="800"
          height="500"
          frameBorder="0"
          scrolling="no"
        />
        <div className="Type">
          {`rec: ${clips[clipIdx].rec_type || "unknown"}`}
        </div>
        <div className="Emote">
          Emotes: {emoteCount}
          <button onClick={addEmote}>Add</button>
        </div>
        <div className="Control">
          <button disabled={clipIdx === 0} onClick={prevClip}>
            Prev
          </button>
          <button disabled={clips.length < clipIdx + 1} onClick={nextClip}>
            Next
          </button>
        </div>
      </div>
    </div>
  );
}

export default App;
