const uuid = require('uuid');

const WebSocket = require('./src/ws');
const config = require('./src/config');
const logger = require('./src/logger');

const FortniteGame = require('./src/game/fortnite');
const ApexGame = require('./src/game/apex');

class Fortwatch {
  constructor() {
    logger.log('FortWatch started');

    this.onMessage = this.onMessage.bind(this);
    this.onUser = this.onUser.bind(this);
    this.onUserGame = this.onUserGame.bind(this);
    this.emitUserState = this.emitUserState.bind(this);
    this.doEmitUserState = this.doEmitUserState.bind(this);
    this.destroyInterval = this.destroyInterval.bind(this);
    this.createInterval = this.createInterval.bind(this);

    this.user = null;
    this.retryTimeout = null;
    this.activeGame = null;
    this.isGameHealthy = false;
    this.healthCheckInterval = null;
    this.userStateInterval = null;
    this.isActive = false; // So we don't randomly start tailing based on /match

    this.ws = new WebSocket(`ws://localhost:${config.WS_PORT}`);
    this.ws.addHandler(this.onMessage);

    this.ws.wsSend({
      url: '/user/me',
      method: 'GET',
      req_id: uuid.v4()
    });

    this.createInterval();
  }

  getMatch(match_id) {
    this.ws.wsSend({
      url: '/match',
      method: 'GET',
      req_id: uuid.v4(),
      match_id: match_id
    });
  }

  onMessage(json) {
    const { url, ack_id, code } = json;

    if (!url) {
      return;
    }

    // filter out loggin otherwise it gets spammy
    if (url === '/user' || url === '/user/me' || url === '/user/game') {
      logger.info("Fortwatch::onMessage url:", url);
    } else {
      // logger.debug("Fortwatch::onMessage url:", url);
    }

    if (url === '/user' || url === '/user/me') {
      if (!json.result || !json.result.length) {
        return;
      }
      const user = json.result[0];
      this.onUser(user)
    } else if (url === '/user/game') {
      if (!ack_id) {
        return;
      }
      if (!json.result || !json.result.length || code !== 200) {
        logger.warn('handleUserGame server not happy', json);
        return;
      }
      const user_game = json.result[0];
      this.onUserGame(user_game);
    } else if (url === '/match') {
      if (!json.result || !json.result.length) {
        return;
      }
      const match = json.result[0];
      this.onMatch(match);
    }
  }

  onMatch(match) {
    const gameName = match.game_id || 'fortnite';

    if (!this.activeGame) {
      this.startGame(gameName);
    } else {
      const currentActiveGameName = this.activeGame.gameName;
      if (currentActiveGameName !== gameName) {
        this.stopActiveGame();
        this.startGame(gameName);
      }
    }
  }

  onUser(user) {
    this.user = user;

    const active_match_id = user.active_match_id;
    const current_platform = user.current_platform || 'pc';

    logger.info('Fortwatch::onUser',
      'active_match_id:', active_match_id,
      'current_platform:', current_platform);

    if (active_match_id && current_platform === 'pc') {
      this.isActive = true;
      this.getMatch(active_match_id);
    } else {
      this.isActive = false;
      this.stopActiveGame();
    }
  }

  onUserGame(user_game) {
    if (!this.activeGame) {
      logger.error('Received user game with no active game running.', user_game);
      return;
    }
    if (!user_game.ended_dttm) {
      this.activeGame.currentGameId = user_game.user_game_id;
    } else {
      this.activeGame.currentGameId = null;
    }
  }

  destroyInterval() {
    if (this.healthCheckInterval) {
      clearInterval(this.healthCheckInterval);
      this.healthCheckInterval = null;
    }

    if (this.userStateInterval) {
      clearInterval(this.userStateInterval);
      this.userStateInterval = null;
    }
  }

  createInterval() {
    if (!this.userStateInterval) {
      this.doEmitUserState();
      this.userStateInterval = setInterval(
        this.doEmitUserState.bind(this), 10000);
    }

    if (!this.healthCheckInterval) {
      this.doGameHealthCheck();
      this.healthCheckInterval = setInterval(
        this.doGameHealthCheck.bind(this), 5000);
    }
  }

  // Consistently pinging the backend, to inform that we're alive
  doEmitUserState() {
    if (this.activeGame && this.isGameHealthy) {
      const gameState = this.activeGame.getGameState();
      this.emitUserState("POST", gameState);
    }
  }

  // The function only emit if game's health status changed
  async doGameHealthCheck() {
    if (!this.activeGame) {
      return;
    }

    const last_health_status = this.isGameHealthy;

    if (this.activeGame) {
      this.isGameHealthy = await this.activeGame.isHealthy();
      if (last_health_status !== this.isGameHealthy) {
        if (this.isGameHealthy) {
          const gameState = this.activeGame.getGameState();
          this.emitUserState("POST", gameState);
        } else {
          this.emitUserState("DELETE");
          this.activeGame.reset();
        }
      }
    }
  }

  onGameStateChange(oldState, newState, userGameId) {
    logger.info('Fortwatch::onGameStateChange', oldState + '->' + newState);
    if (oldState === newState) {
      return;
    }
    if (newState === 'game') {
      this.ws.wsSend({
        req_id: uuid.v4(),
        method: 'POST',
        url: '/user/game',
        started_dttm: '$NOW',
      });
    } else if (newState === 'lobby') {
      this.ws.wsSend({
        req_id: uuid.v4(),
        method: 'PUT',
        url: '/user/game',
        user_game_id: userGameId,
        ended_dttm: '$NOW',
      });
    }
  }

  emitUserState(method, data={}) {
    logger.debug('Fortwatch::emitUserState', method, data);
    const payload = {
      req_id: uuid.v4(),
      url: '/user/state/fortwatch',
      method
    }

    if (method.toUpperCase() === 'POST') {
      payload['expire'] = 15;
      payload['data'] = data;
    }

    this.ws.wsSend(payload);
  }

  startGame(game_name) {
    logger.info('Fortwatch::StartGame', game_name);
    if (!this.isActive) {
      return;
    }
    if (this.activeGame) {
      this.stopActiveGame();
    }

    if (game_name === 'fortnite') {
      this.activeGame = new FortniteGame(this.onGameStateChange.bind(this));
    } else if (game_name === 'apex') {
      this.activeGame = new ApexGame(this.onGameStateChange.bind(this));
    }

    this.activeGame.start();
  }

  stopActiveGame() {
    logger.info('Fortwatch::StopActiveGame');
    if (!this.isActive) {
      return;
    }
    if (!this.activeGame) {
      return;
    }
    this.activeGame.reset();
    this.activeGame.stop();
    this.activeGame = null;
  }
}

const fortwatch = new Fortwatch();
