const path = require('path');
const Game = require("./game");
const logger = require('../logger');


const FORTNITE_PROCESS_NAME = 'FortniteClient-Win64-Shipping.exe';
const LOG_FILE_PATH = path.join(process.env.LOCALAPPDATA, "FortniteGame", "Saved",
                            "Logs", "FortniteGame.log");

const IGNORE_REGEX = /(ParticleSystem\=|Template\:)(\/.*)+/;
const FORTNITE_LOG_REGEX = /\[([0-9]{4}\.[0-9]{2}\.[0-9]{2}\-[0-9]{2}\.[0-9]{2}\.[0-9]{2}\:[0-9]{3,})\]\[.{3}\](.*?)\:\ (.*)$/;
const UI_CHANGE_REGEX = /Display: \[UFortUIManagerWidget_NUI::TransitionToNextState\] Switching states from: (.*) to (.*)$/;
const PLAYLIST_TO_MODE_REGEX = /(solo|duo|squad|respawn)s?/i;

class Fortnite extends Game {
  constructor(onGameStateChange) {
    super({
      gameName: "fortnite",
      processName: FORTNITE_PROCESS_NAME,
      logFilePath: LOG_FILE_PATH,
      onGameStateChange
    })

    this.whitelistedTags = {
      LogFortUI: this.onUIChange.bind(this),
      LogMatchmakingServiceClient: this.onMatchmakingService.bind(this)
    };
  }

  parseDTTM(dttm) {
    const split = dttm.split('-');
    const date = split[0].split('.');
    const time = split[1].split(':')[0].split('.');
    const milli = split[1].split(':')[1];
    const year = date[0];
    const month = date[1] - 1;
    const day = date[2];
    const hour = time[0];
    const minute = time[1];
    const second = time[2];
    const timestamp = new Date(
      Date.UTC(year, month, day, hour, minute, second, milli),
    );
    return timestamp;
  }

  onLine(line) {
    if (!line) {
      return;
    }
    const match = line.match(FORTNITE_LOG_REGEX);
    if (!match) {
      const ignore_match = line.match(IGNORE_REGEX);
      if (!ignore_match) {
        logger.debug('Unable to match line w/ REGEX?', line);
      }
      return;
    }
    const dttm = this.parseDTTM(match[1]);
    const tag = match[2];
    const message = match[3];
    if (tag in this.whitelistedTags) {
      this.whitelistedTags[tag](dttm, message);
    }
  }

  onMatchmakingService(dttm, message) {
    if (message.indexOf('StartMatchmaking') > -1) {
      const parts = message.match(/'(.*?)'/i);

      if (!parts || !parts.length) {
        return;
      }

      // 4664262:0:NAW:playlist_deimos_solo_winter
      const mm_bucket = parts[parts.length - 1];
      const mm_parts = mm_bucket.split(':');

      const bucket_id = mm_parts[0];
      const ignored = mm_parts[1];
      const region = mm_parts[2];
      const newPlaylist = mm_parts[3];

      const oldPlaylist = this.currentPlaylist;
      const oldPlaylistMode = this.currentPlaylistMode;

      const newPlaylistMode = this.parsePlaylistMode(newPlaylist);

      logger.info('Playlist transition:', oldPlaylist, '->', newPlaylist);
      logger.info('CurrentPlaylistMode transition:', oldPlaylistMode, '->',
        newPlaylistMode);

      this.currentPlaylist = newPlaylist;
      this.currentPlaylistMode = newPlaylistMode;
    }
  }

  onUIChange(dttm, message) {
    const state_change = message.match(UI_CHANGE_REGEX);
    if (!state_change) {
      return;
    }
    const prev_state = state_change[1];
    const new_state = state_change[2];
    logger.info(
      `Fortnite transitioned from state ${prev_state} to ${new_state}`,
    );
    if (prev_state === 'Invalid' && new_state === 'Athena') {
      const oldGameState = this.currentGameState;
      this.currentGameState = 'game';
      this.onGameStateChange(oldGameState, this.currentGameState, this.currentGameId);
    } else if (prev_state === 'Invalid' && new_state === 'FrontEnd') {
      const oldGameState = this.currentGameState;
      this.currentGameState = 'lobby';
      this.currentPlaylist = 'none';
      this.currentPlaylistMode = 'none'
      this.onGameStateChange(oldGameState, this.currentGameState, this.currentGameId);
    }
  }

  parsePlaylistMode(playlist) {
    const mode_arr = playlist.match(PLAYLIST_TO_MODE_REGEX);
    let newPlaylistMode;
    if (mode_arr) {
      newPlaylistMode = mode_arr[mode_arr.length - 1];
    } else {
      newPlaylistMode = 'none';
    }
    return newPlaylistMode.toLowerCase();
  }
}

module.exports = Fortnite;
