const processExists = require('process-exists');
const { Tail } = require('tail');
const fs = require('fs');

const logger = require('../logger');

class Game {
  constructor({ gameName, processName, logFilePath, onGameStateChange }) {
    this.gameName = gameName;
    this.processName = processName;
    this.logFilePath = logFilePath;

    const noop = () => {};
    this.onGameStateChange = onGameStateChange || noop;

    this.currentGameId = null;
    this.currentGameState = "lobby";
    this.currentPlaylist = "none";
    this.currentPlaylistMode = "none";

    this.tail = null;
    this.retryTailTimeout = null;

    this.start = this.start.bind(this);
    this.stop = this.stop.bind(this);
    this.startTail = this.startTail.bind(this);
    this.stopTail = this.stopTail.bind(this);
    this.onTailError = this.onTailError.bind(this);
  }

  start() {
    this.startTail();
  }

  stop() {
    this.stopTail();
  }

  startTail() {
    try {
      this.tail = new Tail(this.logFilePath, {
        useWatchFile: true,
        seperator: '\n',
        fromBeginning: false,
        follow: true,
        logger
      });
      // might want flushAtEOF: set to true if you want to force flush of content when end of file is reached.
      // Particularly useful when there's no separator character at the end of the file (default: false).
      this.tail.on('error', this.onTailError.bind(this));
      this.tail.on('line', line => this.onLine(line.trim()));
    } catch (err) {
      this.onTailError(err);
    }
  }

  stopTail() {
    if (this.tail) {
      this.tail.unwatch();
      this.tail = null;
    }
    if (this.retryTailTimeout) {
      clearTimeout(this.retryTailTimeout);
      this.retryTailTimeout = null;
    }
  }

  onTailError(error) {
    if (this.retryTailTimeout) {
      clearTimeout(this.retryTailTimeout);
    }
    this.retryTailTimeout = setTimeout(this.startTail.bind(this), 3000);
  }

  getGameState() {
    return {
      state: this.currentGameState,
      gamestate: this.currentGameState,
      playlist: this.currentPlaylist,
      mode: this.currentPlaylistMode
    }
  }

  reset() {
    const oldState = this.currentGameState;
    const gameId = this.currentGameId;
    this.currentGameId = null;
    this.currentGameState = "lobby";
    this.currentPlaylist = "none";
    this.currentPlaylistMode = "none";
    this.onGameStateChange(oldState, this.currentGameState, gameId);
  }

  async isHealthy() {
    const processRunning = await processExists(this.processName);
    const fileExists = fs.existsSync(this.logFilePath);
    return processRunning && fileExists;
  }

  onLine(line) {
  }
}

module.exports = Game;

