const BeboNodeWS = require('bebo-node-ws');

const logger = require('./logger');

class Websocket {
  constructor(url) {
    this.handlers = [];
    this.sendQueue = [];

    this.addHandler = this.addHandler.bind(this);
    this.wsSend = this.wsSend.bind(this);

    this.onOpen = this.onOpen.bind(this);
    this.onClose = this.onClose.bind(this);
    this.onMessage = this.onMessage.bind(this);
    this.onError = this.onError.bind(this);

    this.client = new BeboNodeWS(url);
    this.client.addEventListener('open', this.onOpen);
    this.client.addEventListener('close', this.onClose);
    this.client.addEventListener('message', this.onMessage);
    this.client.addEventListener('error', this.onError);
  }

  addHandler(handler) {
    this.handlers.push(handler);
  }

  removeHandler(handler) {
    const index = this.handlers.findIndex(h => h === handler);
    if (index < 0) {
      return;
    }
    this.handlers.splice(index, 1);
    logger.debug('Removed handler. Current handlers:', this.handlers);
  }

  onOpen() {
    logger.debug('onOpen');
    if (this.sendQueue.length) {
      logger.info(`flushing sendQueue of ${this.sendQueue.length} messages`);
      for (let json of this.sendQueue) {
        this.wsSend(json);
      }
    }
  }

  wsSend(json) {
    const url = json.url || 'unknown url';
    if (this.client.readyState !== BeboNodeWS.OPEN) {
      logger.info('not sending into websocket as websocket is not open', url);
      this.sendQueue.push(json);
      return;
    }

    let msg;

    try {
      msg = JSON.stringify(json);
    } catch (err) {
      logger.error('Failed to stringify message', json);
    }

    if (!msg) {
      return;
    }

    this.client.send(msg);
  }

  onClose() {
    logger.debug('onClose');
  }

  onError(error) {
    logger.error(`Error in WS. message: ${error.message}, error:`, error.error || error);
  }

  onMessage(msgEvent) {
    const message = msgEvent.data;
    if (message === 'ping') {
      return;
    }

    let json;
    try {
      json = JSON.parse(message);
    } catch (err) {
      logger.error('Failed to parse message from WS', err);
    }
    if (!json) {
      return;
    }
    const { url } = json;
    if (!url) {
      logger.warn('no url?', json);
      return;
    }
    for (let handler of this.handlers) {
      handler(json);
    }
  }
}

module.exports = Websocket;
