
locals {
  # Normalized bucket reference for output bucket, ensuring a created bucket is in the resource graph for usages of its name.
  # Otherwise it may not be created before Terraform attempts to use references to it.
  computed_s3_output_bucket = var.create_s3_output_bucket == 1 ? aws_s3_bucket.output[0].bucket : var.s3_output_bucket

  # Simplify boolean checking for QA jobs
  is_qa_job = var.qa_for_job_name != ""

  # Render the s3 output key prefix to include the optional qa/ suffix
  rendered_s3_output_key = "${var.s3_output_key}/${local.is_qa_job ? "qa/" : ""}"
}

resource "aws_iam_role" "glue" {
  name               = "${var.job_name}-glue-export"
  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Principal": {
        "Service": "glue.amazonaws.com"
      },
      "Action": "sts:AssumeRole"
    }
  ]
}
EOF
}

resource "aws_iam_role_policy_attachment" "glue-default" {
  role       = aws_iam_role.glue.name
  policy_arn = "arn:aws:iam::aws:policy/service-role/AWSGlueServiceRole"
}

resource "aws_iam_role_policy" "glue-kms" {
  name   = "${aws_iam_role.glue.name}-kms"
  role   = aws_iam_role.glue.name
  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Action": [
        "logs:AssociateKmsKey"
      ],
      "Resource": [
        "arn:aws:logs:*:*:/aws-glue/*"
      ]
    }, {
      "Effect": "Allow",
      "Action": [
             "kms:Encrypt",
             "kms:Decrypt",
             "kms:GenerateDataKey"
      ],
      "Resource": [
        "${length(var.s3_output_kms_key_arn) > 0 ? var.s3_output_kms_key_arn : element(concat(aws_kms_key.s3.*.arn, list("")), 0)}"
      ]
    }
  ]
}
EOF
}

resource "aws_iam_role_policy" "glue-script-s3" {
  name   = "${aws_iam_role.glue.name}-script-s3"
  role   = aws_iam_role.glue.name
  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Action": [
        "s3:GetObject*",
        "s3:ListBucket"
      ],
      "Resource": [
        "arn:aws:s3:::${local.computed_s3_script_bucket}/db_exports/${var.job_name}/*",
        "arn:aws:s3:::${local.computed_s3_script_bucket}"
      ]
    }
  ]
}
EOF
}

resource "aws_iam_role_policy" "glue-output" {
  name   = "${aws_iam_role.glue.name}-output"
  role   = aws_iam_role.glue.name
  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Action": [
        "s3:DeleteObject*",
        "s3:GetObject*",
        "s3:PutObject*",
        "s3:ListBucket"
      ],
      "Resource": [
        "arn:aws:s3:::${var.s3_output_bucket}/${local.rendered_s3_output_key}*",
        "arn:aws:s3:::${var.s3_output_bucket}/*_$folder$",
        "arn:aws:s3:::${var.s3_output_bucket}/_scratch/*",
        "arn:aws:s3:::${var.s3_output_bucket}"
      ]
    }
  ]
}
EOF
}

resource "aws_iam_role_policy" "glue-rds" {
  count = var.database_type == "rds" || var.database_type == "redshift" ? 1 : 0
  name  = "${aws_iam_role.glue.name}-rds"
  role  = aws_iam_role.glue.name
  // ssm:GetParameter Resource is wonky to handle parameter names that start with '/'
  //   since the '/' is stripped from the start of the name in the ARN.
  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Action": "ssm:GetParameter",
      "Resource": [
        "arn:aws:ssm:us-west-2:${var.account_number}:parameter${length(element(split("/", var.db_password_parameter_name), 0)) == 0 ? "" : "/"}${var.db_password_parameter_name}",
        "arn:aws:ssm:us-west-2:${var.account_number}:parameter${length(element(split("/", var.api_key_parameter_name), 0)) == 0 ? "" : "/"}${var.api_key_parameter_name}"
      ]
    }, {
      "Effect": "Allow",
      "Action": "kms:Decrypt",
      "Resource": [
        "arn:aws:kms:us-west-2:${var.account_number}:key/${var.db_password_key_id}",
        "arn:aws:kms:us-west-2:${var.account_number}:key/${var.api_key_kms_key_id}"
      ]
    }, {
      "Effect": "Allow",
      "Action": [
        "rds:AddTagsToResource",
        "rds:RestoreDBInstanceFromDBSnapshot",
        "rds:RestoreDBInstanceToPointInTime",
        "rds:DescribeDBClusters",
        "rds:DescribeDBSnapshots",
        "redshift:DescribeClusters"
      ],
      "Resource": [
        "arn:aws:rds:us-west-2:${var.account_number}:db:${var.cluster_name}",
        "arn:aws:rds:us-west-2:${var.account_number}:cluster:${var.cluster_name}",
        "arn:aws:rds:us-west-2:${var.account_number}:snapshot:rds:${var.cluster_name}-*",
        "arn:aws:rds:us-west-2:${var.account_number}:db:${var.job_name}-export-snapshot",
        "arn:aws:rds:us-west-2:${var.account_number}:subgrp:${var.rds_subnet_group}",
        "arn:aws:redshift:us-west-2:${var.account_number}:cluster:${var.cluster_name}"
      ]
    }, {
      "Effect": "Allow",
      "Action": "rds:DescribeDBInstances",
      "Resource": "*"
    }, {
      "Effect": "Allow",
      "Action": [
        "rds:DeleteDBInstance",
        "rds:ModifyDBInstance"
      ],
      "Resource": "arn:aws:rds:us-west-2:${var.account_number}:db:${var.job_name}-export-snapshot"
    }
  ]
}
EOF
}

resource "aws_iam_role_policy" "glue-dynamodb" {
  count  = var.database_type == "dynamodb" ? 1 : 0
  name   = "${aws_iam_role.glue.name}-dynamodb"
  role   = aws_iam_role.glue.name
  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Action": [
        "dynamodb:DescribeTable",
        "dynamodb:Scan"
      ],
      "Resource": ["${join("\", \"", formatlist("arn:aws:dynamodb:us-west-2:%s:table/%s", var.account_number, keys(var.table_config)))}"]
    }, {
      "Effect": "Allow",
      "Action": "ssm:GetParameter",
      "Resource": [
        "arn:aws:ssm:us-west-2:${var.account_number}:parameter${length(element(split("/", var.api_key_parameter_name), 0)) == 0 ? "" : "/"}${var.api_key_parameter_name}"
      ]
    }, {
      "Effect": "Allow",
      "Action": "kms:Decrypt",
      "Resource": [
        "arn:aws:kms:us-west-2:${var.account_number}:key/${var.api_key_kms_key_id}"
      ]
    }
  ]
}
EOF
}

# Needed until security configurations can be managed by Terraform.
resource "aws_iam_role_policy" "glue-security-config" {
  name   = "${aws_iam_role.glue.name}-security-config"
  role   = aws_iam_role.glue.name
  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Action": [
        "glue:CreateSecurityConfiguration",
        "glue:GetSecurityConfiguration",
        "glue:GetJob",
        "glue:UpdateJob"
      ],
      "Resource": "*"
    }, {
      "Effect": "Allow",
      "Action": [
        "iam:PassRole"
      ],
      "Resource": "${aws_iam_role.glue.arn}"
    }
  ]
}
EOF
}

resource "aws_iam_role_policy" "glue-tahoe-api" {
  name   = "${aws_iam_role.glue.name}-glue-tahoe-api"
  role   = aws_iam_role.glue.name
  count  = length(var.tahoe_producer_name) > 0 ? 1 : 0
  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": [
          "sts:AssumeRole"
      ],
      "Effect": "Allow",
      "Resource": [
          "${var.tahoe_producer_role_arn}"
      ]
    }
  ]
}
EOF
}

provider "null" {}

// Validation hack until Terraform 0.13 from https://github.com/hashicorp/terraform/issues/2847#issuecomment-343622460
resource "null_resource" "check_qa_s3_bucket" {
  count = local.is_qa_job && tonumber(var.create_s3_output_bucket) == 1 ? "x" : 0 # Error: Must reuse production output bucket for QA jobs
}

resource "null_resource" "check_qa_kms_key" {
  count = local.is_qa_job && var.s3_output_kms_key_arn == "" ? "x" : 0 # Error: Must set s3_output_kms_key_arn for QA jobs
}

resource "null_resource" "check_qa_job_name" {
  count = local.is_qa_job && var.qa_for_job_name == var.job_name ? "x" : 0 # Error: Must set a new job_name for QA jobs
}

resource "aws_s3_bucket" "output" {
  count = var.create_s3_output_bucket

  bucket = var.s3_output_bucket
  acl    = "private"

  tags = {
    Service     = var.service
    Environment = var.env
  }

  lifecycle_rule {
    enabled = true
    expiration {
      days = 90
    }
  }
}

resource "aws_s3_bucket_policy" "tahoe-api-access" {
  bucket     = var.s3_output_bucket
  count      = length(var.tahoe_producer_name) > 0 && var.create_s3_output_bucket == 1 ? 1 : 0
  depends_on = [aws_s3_bucket.output]

  policy = <<POLICY
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Sid": "TahoeAPIS3Access",
      "Effect": "Allow",
      "Principal": {
        "AWS": "${var.tahoe_producer_role_arn}"
      },
      "Action": [
        "s3:ListBucket",
        "s3:GetObject"
      ],
      "Resource": [
        "arn:aws:s3:::${var.s3_output_bucket}",
        "arn:aws:s3:::${var.s3_output_bucket}/*"
      ]
    }
  ]
}
POLICY
}


resource "aws_s3_bucket" "glue-scripts" {
  count  = var.create_s3_script_bucket
  bucket = var.s3_script_bucket
  acl    = "private"

  tags = {
    Service     = var.service
    Environment = var.env
  }
}

resource "aws_s3_bucket_object" "jsonschema-lib" {
  bucket = var.s3_script_bucket
  key    = "db_exports/${var.job_name}/glue-libs/jsonschema-3.2.0-py2.py3-none-any.whl"
  source = "${path.module}/resources/jsonschema-3.2.0-py2.py3-none-any.whl"

  tags = {
    Service     = var.service
    Environment = var.env
  }
}

resource "aws_glue_job" "watcher" {
  count        = var.database_type == "dynamodb" ? 0 : 1
  name         = "${var.job_name}-export-watcher"
  role_arn     = aws_iam_role.glue.arn
  max_capacity = 0.0625
  command {
    name            = "pythonshell"
    script_location = "s3://${var.s3_script_bucket}/db_exports/${var.job_name}/watcher.py"
    python_version  = "3"
  }
  timeout = 1440 # 24-hour timeout
  default_arguments = {
    "--enable-metrics"                   = ""
    "--use_existing_cluster"             = "0"
    "--preserve_cluster"                 = "0"
    "--ts"                               = "0"
    "--skip_db_export"                   = "0"
    "--fail_on_error"                    = var.fail_on_error
    "--use_latest_snapshot"              = "0"
    "--enable-continuous-cloudwatch-log" = "true"
    "--enable-continuous-log-filter"     = "true"
    "--extra-py-files"                   = "s3://${aws_s3_bucket_object.jsonschema-lib.bucket}/${aws_s3_bucket_object.jsonschema-lib.key}"
  }
  glue_version = "1.0"

  tags = {
    Service     = var.service
    Environment = var.env
  }
}

resource "aws_glue_job" "watcher_dynamodb" {
  count        = var.database_type == "dynamodb" ? 1 : 0
  name         = "${var.job_name}-export-watcher"
  role_arn     = aws_iam_role.glue.arn
  max_capacity = 0.0625
  command {
    name            = "pythonshell"
    script_location = "s3://${var.s3_script_bucket}/db_exports/${var.job_name}/watcher.py"
    python_version  = "3"
  }
  timeout = 1440 # 24-hour timeout
  default_arguments = {
    "--enable-metrics"                   = ""
    "--use_existing_cluster"             = "0"
    "--preserve_cluster"                 = "0"
    "--ts"                               = "0"
    "--skip_db_export"                   = "0"
    "--fail_on_error"                    = var.fail_on_error
    "--enable-continuous-cloudwatch-log" = "true"
    "--enable-continuous-log-filter"     = "true"
    "--extra-py-files"                   = "s3://${aws_s3_bucket_object.jsonschema-lib.bucket}/${aws_s3_bucket_object.jsonschema-lib.key}"
  }
  glue_version = "1.0"

  tags = {
    Service     = var.service
    Environment = var.env
  }
}

resource "aws_glue_job" "table_export_dynamodb" {
  count    = var.database_type == "dynamodb" ? 1 : 0
  name     = "${var.job_name}-export-table"
  role_arn = aws_iam_role.glue.arn
  command {
    script_location = "s3://${var.s3_script_bucket}/${aws_s3_bucket_object.table_job_script.key}"
    python_version  = "3"
  }
  timeout = 1440 # 24-hour timeout
  execution_property {
    max_concurrent_runs = var.max_concurrent_runs
  }
  default_arguments = {
    "--enable-metrics"                   = ""
    "--ts"                               = ""
    "--table_name"                       = ""
    "--fail_on_error"                    = var.fail_on_error
    "--enable-continuous-cloudwatch-log" = "true"
    "--enable-continuous-log-filter"     = "true"
  }
  security_configuration = aws_glue_security_configuration.export.name
  glue_version           = "1.0"

  tags = {
    Service     = var.service
    Environment = var.env
  }
}

resource "aws_glue_job" "table_export" {
  count    = var.database_type == "dynamodb" ? 0 : 1
  name     = "${var.job_name}-export-table"
  role_arn = aws_iam_role.glue.arn
  command {
    script_location = "s3://${var.s3_script_bucket}/${aws_s3_bucket_object.table_job_script.key}"
    python_version  = "3"
  }
  timeout = 1440 # 24-hour timeout
  execution_property {
    max_concurrent_runs = min(length(var.table_config), var.max_concurrent_runs)
  }
  connections = [element(concat(aws_glue_connection.job-dummy.*.name, list("")), 0)]
  default_arguments = {
    "--fail_on_error"                    = var.fail_on_error
    "--enable-metrics"                   = ""
    "--enable-continuous-cloudwatch-log" = "true"
    "--enable-continuous-log-filter"     = "false"
  }
  security_configuration = aws_glue_security_configuration.export.name
  glue_version           = "1.0"

  tags = {
    Service     = var.service
    Environment = var.env
  }
}

resource "aws_glue_security_configuration" "export" {
  name = "${var.job_name}-export"

  encryption_configuration {
    cloudwatch_encryption {
      kms_key_arn                = aws_kms_key.cloudwatch.arn
      cloudwatch_encryption_mode = "SSE-KMS"
    }

    job_bookmarks_encryption {
      job_bookmarks_encryption_mode = "DISABLED"
    }

    s3_encryption {
      kms_key_arn        = length(var.s3_output_kms_key_arn) > 0 ? var.s3_output_kms_key_arn : element(concat(aws_kms_key.s3.*.arn, list("")), 0)
      s3_encryption_mode = "SSE-KMS"
    }
  }
}

# Used to set security group on job
resource "aws_glue_connection" "job-dummy" {
  count = var.database_type == "dynamodb" ? 0 : 1
  connection_properties = {
    JDBC_CONNECTION_URL = "jdbc:postgresql://example.com/dummy"
    PASSWORD            = "dummy"
    USERNAME            = "dummy"
  }

  name = "${var.job_name}-dummy"
  physical_connection_requirements {
    availability_zone      = var.availability_zone
    security_group_id_list = [aws_security_group.glue[0].id]
    subnet_id              = var.subnet_id
  }
}

resource "aws_security_group" "glue" {
  name        = "${var.job_name}-glue-export"
  count       = var.database_type == "dynamodb" ? 0 : 1
  description = "Allow DB export via Glue"
  vpc_id      = var.vpc_id

  ingress {
    from_port = 0
    to_port   = 65535
    protocol  = "tcp"
    self      = true
  }

  egress {
    from_port   = 0
    to_port     = 0
    protocol    = "-1"
    cidr_blocks = ["0.0.0.0/0"]
  }

  tags = {
    Service     = var.service
    Environment = var.env
  }
}

resource "aws_glue_trigger" "export" {
  name     = "${var.job_name}-export"
  count    = local.is_qa_job ? 0 : 1
  type     = "SCHEDULED"
  schedule = "cron(${var.trigger_schedule})"
  actions {
    job_name = element(concat(aws_glue_job.watcher.*.name, aws_glue_job.watcher_dynamodb.*.name), 0)
  }

  tags = {
    Service     = var.service
    Environment = var.env
  }
}

data "template_file" "tahoe_api_kms_policy" {
  template = <<END_OF_STRING
{
  "Sid": "Enable Tahoe API Role to decrypt",
  "Effect": "Allow",
  "Principal": {
      "AWS": "$${tahoe_producer_role_arn}"
  },
  "Action": "kms:Decrypt",
  "Resource": "*"
}
END_OF_STRING

  vars = {
    tahoe_producer_role_arn = var.tahoe_producer_role_arn
  }
}

resource "aws_kms_key" "s3" {
  description = "Encrypt s3 data exports from ${var.job_name}"
  count       = var.s3_output_kms_key_arn == "" ? 1 : 0
  policy      = <<EOF
{
  "Version": "2012-10-17",
  "Id": "key-default-1",
  "Statement": [
    {
      "Sid": "Enable IAM User Permissions",
      "Effect": "Allow",
      "Principal": {
        "AWS": "arn:aws:iam::${var.account_number}:root"
      },
      "Action": "kms:*",
      "Resource": "*"
    }${length(var.tahoe_producer_name) > 0 ? ",${data.template_file.tahoe_api_kms_policy.rendered}" : ""}
  ]
}
EOF

  tags = {
    Service     = var.service
    Environment = var.env
  }
}

resource "aws_kms_alias" "s3" {
  name          = "alias/${var.job_name}-export-s3"
  count         = var.s3_output_kms_key_arn == "" ? 1 : 0
  target_key_id = aws_kms_key.s3[0].key_id
}

resource "aws_kms_key" "cloudwatch" {
  description = "Encrypt cloudwatch logs for exports from ${var.job_name}"
  policy      = <<EOF
{
  "Version": "2012-10-17",
  "Id": "key-default-1",
  "Statement": [
    {
      "Sid": "Enable IAM User Permissions",
      "Effect": "Allow",
      "Principal": {
        "AWS": "arn:aws:iam::${var.account_number}:root"
      },
      "Action": "kms:*",
      "Resource": "*"
    }, {
      "Effect": "Allow",
      "Principal": {
        "Service": "logs.us-west-2.amazonaws.com",
        "AWS": "${aws_iam_role.glue.arn}"
      },
      "Action": [
             "kms:Encrypt*",
             "kms:Decrypt*",
             "kms:ReEncrypt*",
             "kms:GenerateDataKey*",
             "kms:Describe*"
      ],
      "Resource": "*"
    }
  ]
}
EOF

  tags = {
    Service     = var.service
    Environment = var.env
  }
}

resource "aws_kms_alias" "cloudwatch" {
  name          = "alias/${var.job_name}-export-cloudwatch"
  target_key_id = aws_kms_key.cloudwatch.key_id
}

resource "aws_cloudwatch_event_rule" "error" {
  name        = "${var.job_name}-export-error"
  description = "Errors running Glue export job ${var.job_name}"

  event_pattern = <<EOF
{
  "detail-type": ["Glue Job State Change"],
  "detail": {
    "state": ["FAILED"],
    "jobName": ["${element(concat(aws_glue_job.watcher.*.name, aws_glue_job.watcher_dynamodb.*.name), 0)}"]
  }
}
EOF

  tags = {
    Service     = var.service
    Environment = var.env
  }
}

resource "aws_cloudwatch_event_target" "error-sns" {
  rule      = aws_cloudwatch_event_rule.error.name
  target_id = "${var.job_name}-export-error"
  arn       = "arn:aws:sns:us-west-2:${var.account_number}:${var.error_sns_topic_name}"
  count     = local.is_qa_job ? 0 : 1
}

resource "aws_iam_role" "data_infra_audit" {
  name               = "${var.job_name}-di-db-s3-glue-audit"
  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Principal": {
        "AWS": "arn:aws:iam::260891137513:root"
      },
      "Action": "sts:AssumeRole"
    }
  ]
}
EOF
}

resource "aws_iam_role_policy" "data_infra_audit" {
  name   = "${aws_iam_role.data_infra_audit.name}-main"
  role   = aws_iam_role.data_infra_audit.name
  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Action": [
        "s3:ListBucket"
      ],
      "Resource": [
        "arn:aws:s3:::${var.s3_output_bucket}",
        "arn:aws:s3:::${var.s3_script_bucket}"
      ]
    }, {
      "Effect": "Allow",
      "Action": [
        "s3:GetObject"
      ],
      "Resource": [
        "arn:aws:s3:::${var.s3_script_bucket}/db_exports/${var.job_name}/*"
      ]
    }, {
      "Effect": "Allow",
      "Action": [
        "logs:GetLogEvents"
      ],
      "Resource": [
        "arn:aws:logs:*:*:/aws-glue/*"
      ]
    }, {
      "Effect": "Allow",
      "Action": [
        "sns:GetTopicAttributes",
        "sns:ListSubscriptionsByTopic",
        "sns:ListTagsForResource"
      ],
      "Resource": [
        "arn:aws:sns:us-west-2:${var.account_number}:${var.error_sns_topic_name}"
      ]
    }, {
      "Effect": "Allow",
      "Action": [
        "cloudwatch:GetMetricData",
        "cloudwatch:GetMetricStatistics",
        "cloudwatch:ListMetrics",
        "glue:BatchGetJobs",
        "glue:GetJob",
        "glue:GetJobRun",
        "glue:GetJobRuns",
        "glue:GetJobs",
        "glue:GetTags",
        "glue:GetTriggers",
        "glue:ListJobs",
        "glue:ListTriggers",
        "logs:DescribeLogGroups",
        "logs:DescribeLogStreams",
        "s3:ListAllMyBuckets",
        "sns:ListTopics",
        "tag:GetResources"
      ],
      "Resource": "*"
    }
  ]
}
EOF
}

resource "aws_iam_role_policy" "data_infra_audit_rds" {
  count  = var.database_type == "rds" || var.database_type == "redshift" ? 1 : 0
  name   = "${aws_iam_role.data_infra_audit.name}-rds"
  role   = aws_iam_role.data_infra_audit.name
  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Action": [
        "rds:DescribeDBClusterSnapshots",
        "rds:DescribeDBClusters",
        "rds:DescribeDBInstanceAutomatedBackups",
        "rds:DescribeDBInstances",
        "rds:DescribeDBSnapshots",
        "rds:DescribeEvents",
        "redshift:DescribeClusters"
      ],
      "Resource": "*"
    }
  ]
}
EOF
}

resource "aws_iam_role_policy" "data_infra_audit_dynamo" {
  count  = var.database_type == "dynamodb" ? 1 : 0
  name   = "${aws_iam_role.data_infra_audit.name}-dynamo"
  role   = aws_iam_role.data_infra_audit.name
  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Action": [
        "dynamodb:DescribeTable"
      ],
      "Resource": ["${join("\", \"", formatlist("arn:aws:dynamodb:us-west-2:%s:table/%s", var.account_number, keys(var.table_config)))}"]
    }, {
      "Effect": "Allow",
      "Action": [
        "dynamodb:ListTables"
      ],
      "Resource": "*"
    }
  ]
}
EOF
}


resource "aws_iam_role" "data_infra_import" {
  name               = "${var.job_name}-data-infra-import"
  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Principal": {
        "AWS": "arn:aws:iam::673385534282:root"
      },
      "Action": "sts:AssumeRole"
    }
  ]
}
EOF
}

resource "aws_iam_role_policy" "data_infra_import" {
  name   = "${aws_iam_role.data_infra_import.name}-data-infra-import"
  role   = aws_iam_role.data_infra_import.name
  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Action": "kms:Decrypt",
      "Resource": "${length(var.s3_output_kms_key_arn) > 0 ? var.s3_output_kms_key_arn : element(concat(aws_kms_key.s3.*.arn, list("")), 0)}"
    }, {
      "Effect": "Allow",
      "Action": [
        "s3:GetObject*",
        "s3:ListBucket"
      ],
      "Resource": [
        "arn:aws:s3:::${var.s3_output_bucket}/${local.rendered_s3_output_key}*",
        "arn:aws:s3:::${var.s3_output_bucket}/*_$folder$",
        "arn:aws:s3:::${var.s3_output_bucket}"
      ]
    }
  ]
}
EOF
}

output "s3_kms_key" {
  value = length(var.s3_output_kms_key_arn) > 0 ? var.s3_output_kms_key_arn : element(concat(aws_kms_key.s3.*.arn, list("")), 0)
}

output "s3_output_bucket" {
  value = var.s3_output_bucket
}

output "s3_output_prefix" {
  value = local.rendered_s3_output_key
}

output "glue_role" {
  value = aws_iam_role.glue.arn
}

output "data_infra_role" {
  value = aws_iam_role.data_infra_import.arn
}
