resource "aws_s3_bucket_object" "watcher_job_script_rds" {
  count   = var.database_type == "rds" || var.database_type == "redshift" ? 1 : 0
  bucket  = local.computed_s3_script_bucket
  key     = "db_exports/${var.job_name}/watcher.py"
  content = <<EOF

${data.template_file.common_code.rendered}

args = getResolvedOptions(sys.argv, ['preserve_cluster', 'use_existing_cluster', 'ts', 'skip_db_export', 'use_latest_snapshot', 'fail_on_error'])

now, ts = get_ts(args['ts'])
fail_on_error = args['fail_on_error']
logger.info('Going to write to ts %s', ts)
subjob_name = '${element(concat(aws_glue_job.table_export.*.name, list("")), 0)}'

session = boto3.Session(region_name='us-west-2')
glue = patched_glue_client(session)

tahoe_session, producer_api_key = None, None

if '${var.tahoe_producer_name}':
    tahoe_import_id = datetime.datetime.utcnow().strftime('%Y%m%dT%H%M%S.%f')
    tahoe_session = get_tahoe_session(session, tahoe_import_id)

    ssm = session.client('ssm', region_name='us-west-2')
    ssm_response = ssm.get_parameter(Name='${var.api_key_parameter_name}', WithDecryption=True)
    producer_api_key = ssm_response['Parameter']['Value']

validate_config(tahoe_session, producer_api_key)

snapshot_cluster_name = '${var.job_name}-export-snapshot'
rds = session.client('rds')

def wait_for_status(desired_status):
    while True:
        try:
            status = rds.describe_db_instances(
                DBInstanceIdentifier=snapshot_cluster_name)['DBInstances'][0]['DBInstanceStatus']
        except rds.exceptions.DBInstanceNotFoundFault:
            if desired_status == 'deleted':
                return
            continue
        if desired_status == status:
            break
        elif 'failed' == status:
            raise RuntimeError('Cluster failed, current status: "{}" and not "{}"'.format(status, desired_status))
        else:
            # get this into the logs in case it doesn't come up for some odd reason
            logger.info('sleeping due to status: "%s". waiting for "%s"', status, desired_status)
            time.sleep(60)

def restore_db_instance_from_latest_snapshot(db_instance_identifier):
    try:
        snapshots = rds.describe_db_snapshots(
            DBInstanceIdentifier=db_instance_identifier,
            SnapshotType='automated')['DBSnapshots']
        latest = snapshots[0]
        for snap in snapshots:
            if snap['SnapshotCreateTime'] > latest['SnapshotCreateTime']:
                latest = snap
        logger.info('Restoring snapshot %s to target %s',
            latest['DBSnapshotIdentifier'],
            snapshot_cluster_name)
        instance_class = '${var.rds_instance_class_override}'
        if instance_class:
            rds.restore_db_instance_from_db_snapshot(
                DBSnapshotIdentifier=latest['DBSnapshotIdentifier'],
                DBInstanceIdentifier=snapshot_cluster_name,
                DBInstanceClass=instance_class,
                DBSubnetGroupName='${var.rds_subnet_group}',
                Tags=[{'Key': 'project', 'Value': 'db-s3-glue'}])
        else:
            rds.restore_db_instance_from_db_snapshot(
                DBSnapshotIdentifier=latest['DBSnapshotIdentifier'],
                DBInstanceIdentifier=snapshot_cluster_name,
                DBSubnetGroupName='${var.rds_subnet_group}',
                Tags=[{'Key': 'project', 'Value': 'db-s3-glue'}])
    except Exception:
        logger.exception('Unable to restore_db_instance_from_latest_snapshot')
        raise

preserve_cluster = int(args['preserve_cluster']) or ${var.skip_snapshot}
create_cluster = False
restore_from_snapshot = int(args['use_latest_snapshot']) or ${var.use_latest_snapshot}

if not ${var.skip_snapshot}:
    try:
        resp = rds.describe_db_instances(DBInstanceIdentifier=snapshot_cluster_name)
    except rds.exceptions.DBInstanceNotFoundFault:
        create_cluster = True
    else:
        if not int(args['use_existing_cluster']):
            rds.delete_db_instance(DBInstanceIdentifier=snapshot_cluster_name, SkipFinalSnapshot=True)
            wait_for_status('deleted')
            create_cluster = True

try:
    if not ${var.skip_snapshot}:
        if create_cluster:
            if restore_from_snapshot:
                restore_db_instance_from_latest_snapshot('${var.cluster_name}')
            else:
                restore_time = rds.describe_db_instances(
                    DBInstanceIdentifier='${var.cluster_name}')['DBInstances'][0]['LatestRestorableTime']
                restore_time = restore_time.replace(tzinfo=None)
                restore_time = min(restore_time, now).isoformat() + 'Z'
                logger.info('Restoring source %s to target  %s at %s',
                    '${var.cluster_name}',
                    snapshot_cluster_name,
                    restore_time)
                instance_class = '${var.rds_instance_class_override}'
                if instance_class:
                    rds.restore_db_instance_to_point_in_time(
                        SourceDBInstanceIdentifier='${var.cluster_name}',
                        TargetDBInstanceIdentifier=snapshot_cluster_name,
                        RestoreTime=restore_time,
                        DBInstanceClass=instance_class,
                        DBSubnetGroupName='${var.rds_subnet_group}',
                        Tags=[{'Key': 'project', 'Value': 'db-s3-glue'}])
                else:
                    rds.restore_db_instance_to_point_in_time(
                        SourceDBInstanceIdentifier='${var.cluster_name}',
                        TargetDBInstanceIdentifier=snapshot_cluster_name,
                        RestoreTime=restore_time,
                        DBSubnetGroupName='${var.rds_subnet_group}',
                        Tags=[{'Key': 'project', 'Value': 'db-s3-glue'}])

        wait_for_status('available')

        # now change security group and wait
        instance_class = '${var.rds_instance_class_override}'
        logger.info('Modifying security group and cert authority for %s', snapshot_cluster_name)
        rds.modify_db_instance(
            DBInstanceIdentifier=snapshot_cluster_name,
            BackupRetentionPeriod=0,  # means no backups for these dbs
            ApplyImmediately=True,
            # Needed to silence alerts on accounts. Can remove when it is the default.
            CACertificateIdentifier='rds-ca-2019',
            VpcSecurityGroupIds=['${aws_security_group.glue[0].id}'])

    process_all_tables(tahoe_import_id, tahoe_session, producer_api_key, int(args['skip_db_export']), {'--fail_on_error': fail_on_error})

except Exception:
    if not preserve_cluster:
        try:
            rds.delete_db_instance(DBInstanceIdentifier=snapshot_cluster_name, SkipFinalSnapshot=True)
            wait_for_status('deleted')
        except Exception:
            logger.exception('Unable to delete: %s')
    raise
if not preserve_cluster:
    rds.delete_db_instance(DBInstanceIdentifier=snapshot_cluster_name, SkipFinalSnapshot=True)
    wait_for_status('deleted')

EOF
}
