package cloudwatch

import (
	"context"
	"time"

	log "github.com/sirupsen/logrus"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/cloudwatch"
)

const (
	Namespace = "Marionette/Crawler"

	UnitCount = "Count"

	cloudwatchTimeout = 5 * time.Second
)

var (
	sess = session.Must(session.NewSessionWithOptions(session.Options{
		SharedConfigState: session.SharedConfigEnable,
	}))
	cloudwatchSvc = cloudwatch.New(sess)
)

func SendMetric(metricName string, value float64, unit, platform string) {
	input := &cloudwatch.PutMetricDataInput{MetricData: []*cloudwatch.MetricDatum{
		&cloudwatch.MetricDatum{
			MetricName: aws.String(metricName),
			Unit:       aws.String(unit),
			Value:      aws.Float64(value),
			Dimensions: []*cloudwatch.Dimension{
				&cloudwatch.Dimension{
					Name:  aws.String("Platform"),
					Value: aws.String(platform),
				},
			},
		},
	}, Namespace: aws.String(Namespace)}

	ctx, cancel := context.WithDeadline(context.Background(), time.Now().Add(cloudwatchTimeout))
	defer cancel()

	_, err := cloudwatchSvc.PutMetricDataWithContext(ctx, input)
	if err != nil {
		log.Warnf("send metric to CloudWatch failed: [%s]. MetricName: [%s], Platform: [%s]", err.Error(), metricName, platform)
	}
}
