package afreeca

import (
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
	"net/url"
	"strconv"
	"strings"
	"sync"
	"time"

	crawler "code.justin.tv/esports-exp/marionette/pkg/crawler/base"
	"code.justin.tv/esports-exp/marionette/pkg/model"
	"code.justin.tv/esports-exp/marionette/pkg/util"
	"github.com/abadojack/whatlanggo"
)

const (
	categoryAPI = "http://www.afreecatv.com/main/main_gamecategory_viewer.js"
	streamAPI   = "https://live.afreecatv.com/api/main_broad_list_api.php"
	pageLimit   = 100
	pageSize    = 60
	batchSize   = 8
)

type AfreecaCrawler struct {
	crawler.BaseCrawler

	categoryMap map[string]string
}

func NewCrawler() (*AfreecaCrawler, error) {
	baseCrawler, err := crawler.NewBaseCrawler("afreeca")
	if err != nil {
		return nil, err
	}

	return &AfreecaCrawler{
		*baseCrawler,
		make(map[string]string),
	}, nil
}

type StreamResponse struct {
	TotalCnt interface{} `json:"total_cnt,string"`
	Cnt      int         `json:"cnt"`
	Broad    []struct {
		BroadNo         string `json:"broad_no"`
		ParentBroadNo   string `json:"parent_broad_no"`
		UserID          string `json:"user_id"`
		UserNick        string `json:"user_nick"`
		BroadTitle      string `json:"broad_title"`
		BroadThumb      string `json:"broad_thumb"`
		BroadStart      string `json:"broad_start"`
		BroadGrade      string `json:"broad_grade"`
		BroadBps        string `json:"broad_bps"`
		BroadResolution string `json:"broad_resolution"`
		VisitBroadType  string `json:"visit_broad_type"`
		BroadType       string `json:"broad_type"`
		StationName     string `json:"station_name"`
		BroadMemo       string `json:"broad_memo"`
		CurrentViewCnt  string `json:"current_view_cnt"`
		MCurrentViewCnt string `json:"m_current_view_cnt"`
		AllowedViewCnt  string `json:"allowed_view_cnt"`
		IsPassword      string `json:"is_password"`
		Rank            string `json:"rank"`
		BroadCateNo     string `json:"broad_cate_no"`
		TotalViewCnt    string `json:"total_view_cnt"`
		PcViewCnt       string `json:"pc_view_cnt"`
		MobileViewCnt   string `json:"mobile_view_cnt"`
	} `json:"broad"`
	Time int `json:"time"`
}

type CategoryResponse struct {
	RESULT int    `json:"RESULT"`
	MSG    string `json:"MSG"`
	DATA   []struct {
		BroadCateNo   string `json:"broad_cate_no"`
		CateName      string `json:"cate_name"`
		CateNameThTh  string `json:"cate_name_th_th"`
		CateNameJaJp  string `json:"cate_name_ja_jp"`
		CateNameEnUs  string `json:"cate_name_en_us"`
		CateNameZhCn  string `json:"cate_name_zh_cn"`
		CateNameZhTw  string `json:"cate_name_zh_tw"`
		CatePcImg     string `json:"cate_pc_img"`
		CateMobileImg string `json:"cate_mobile_img"`
		Viewer        string `json:"viewer"`
	} `json:"DATA"`
}

func (c *AfreecaCrawler) getStreams(pageNum int) (*StreamResponse, error) {
	req, err := http.NewRequest("GET", streamAPI, nil)
	if err != nil {
		return nil, err
	}
	query := &url.Values{}
	query.Add("selectType", "action")
	query.Add("selectValue", "all")
	query.Add("orderType", "view_cnt")
	query.Add("pageNo", strconv.Itoa(pageNum))
	req.URL.RawQuery = query.Encode()

	req.Header.Add("User-Agent", util.GetRandomUserAgent())
	req.Header.Add("Accept", "*/*")
	req.Header.Add("Sec-Fetch-Site", "same-site")
	req.Header.Add("Sec-Fetch-Mode", "no-cors")
	req.Header.Add("Referer", "https://www.afreecatv.com/")
	req.Header.Add("Accept-Language", "en-US,en;q=0.9")

	resp, err := c.HTTPClient.Do(req)
	if err != nil {
		return nil, fmt.Errorf("getStreams failed to fetch: %s", err.Error())
	}
	defer resp.Body.Close()

	if resp.StatusCode != 200 {
		return nil, fmt.Errorf("Request failed: %d %s", resp.StatusCode, resp.Status)
	}

	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return nil, err
	}
	jsonBody := strings.Trim(string(body), "();") // remove brackets

	result := StreamResponse{}
	err = json.Unmarshal([]byte(jsonBody), &result)
	if err != nil {
		return nil, fmt.Errorf("getStreams failed to decode: %s", err.Error())
	}
	return &result, nil
}

func (c *AfreecaCrawler) processStreams(streams *StreamResponse, timestamp int64, results chan<- model.ChannelInfo) {
	for _, stream := range streams.Broad {
		ccv, err := strconv.Atoi(stream.TotalViewCnt)
		if err != nil {
			c.Log.Warningf("Failed to parse ccv: %s %v", stream.TotalViewCnt, err)
			ccv = 0
		}

		game, ok := c.categoryMap[strings.TrimLeft(stream.BroadCateNo, "0")]
		if !ok {
			game = "Other"
		}

		var lang string
		if langInfo := whatlanggo.Detect(stream.BroadTitle); langInfo.Confidence > 0.5 {
			lang = langInfo.Lang.String()
		}

		results <- model.ChannelInfo{
			ChannelID:    stream.UserID,
			ChannelName:  stream.UserNick,
			ChannelTitle: stream.BroadTitle,
			CountryCode:  "", // Unknown
			Game:         game,
			Language:     lang, // Heuristic
			Platform:     c.Platform(),
			Ccv:          int64(ccv),
			TimeCrawled:  timestamp,
		}
	}
}

func (c *AfreecaCrawler) getCategories() error {
	req, err := http.NewRequest("GET", categoryAPI, nil)
	if err != nil {
		return err
	}

	req.Header.Add("Accept", "application/json, text/javascript, */*; q=0.01")
	req.Header.Add("Referer", "http://www.afreecatv.com/?hash=game")
	req.Header.Add("X-Requested-With", "XMLHttpRequest")
	req.Header.Add("User-Agent", util.GetRandomUserAgent())
	req.Header.Add("Content-Type", "application/x-www-form-urlencoded;charset=utf-8")

	resp, err := c.HTTPClient.Do(req)
	if err != nil {
		return err
	}
	defer resp.Body.Close()
	if resp.StatusCode != 200 {
		return fmt.Errorf("afreeca Request failed: %d %s", resp.StatusCode, resp.Status)
	}

	result := CategoryResponse{}
	err = json.NewDecoder(resp.Body).Decode(&result)
	if err != nil {
		return err
	}

	for _, cate := range result.DATA {
		c.categoryMap[cate.BroadCateNo] = cate.CateNameEnUs
	}

	c.Log.Infof("Got categories: %v", c.categoryMap)
	return nil
}

func (c *AfreecaCrawler) Crawl() error {
	timestamp := time.Now().Unix()
	err := c.getCategories()
	if err != nil {
		return fmt.Errorf("Crawl failed to get categories: %s", err.Error())
	}

	results := make(chan model.ChannelInfo, pageLimit*pageSize)
	done := make(chan bool)

	go func() {
		c.TrackChannelInfo(results, 0)
		done <- true
	}()

	for i := 1; i <= pageLimit; i++ {
		result, err := c.getStreams(i)
		if err != nil {
			c.Log.Errorf("Crawl failed to get live rooms %d: %s", i, err.Error())
			break
		}
		c.processStreams(result, timestamp, results)

		if result.Cnt == 0 {
			c.Log.Infof("End of feed")
			break
		}
	}

	allDone := false
	for i := 0; i < int(pageLimit/batchSize) && !allDone; i++ {
		wg := &sync.WaitGroup{}
		for j := 0; j < batchSize && !allDone; j++ {
			wg.Add(1)
			go func(pageIndex int) {
				defer wg.Done()
				result, err := c.getStreams(pageIndex)
				if err != nil {
					c.Log.Errorf("Crawl failed to get live rooms %d: %s", pageIndex, err.Error())
					return
				}
				if result.Cnt == 0 {
					allDone = true
				}
				c.processStreams(result, timestamp, results)
			}(i*batchSize + j + 1)
		}
		wg.Wait()
	}
	if !allDone {
		c.Log.Warnf("fetching too many pages")
	}

	close(results)
	<-done

	return nil
}
