package base

import (
	"context"
	"fmt"
	"net/http"
	"net/url"
	"os"
	"time"

	"code.justin.tv/common/spade-client-go/spade"
	"code.justin.tv/esports-exp/marionette/pkg/alert"
	"code.justin.tv/esports-exp/marionette/pkg/cloudwatch"
	"code.justin.tv/esports-exp/marionette/pkg/config"
	"code.justin.tv/esports-exp/marionette/pkg/model"
	proxyclient "code.justin.tv/esports-exp/marionette/pkg/proxy/client"
	"github.com/davecgh/go-spew/spew"
	"github.com/motemen/go-loghttp"
	"github.com/sirupsen/logrus"
)

const batchSize = 32

type HTTPClient interface {
	Do(req *http.Request) (*http.Response, error)
}

type BaseCrawler struct {
	platform    string
	Log         *logrus.Entry
	SpadeClient spade.Client
	HTTPClient  HTTPClient
}

func NewBaseCrawler(platform string) (*BaseCrawler, error) {
	spadeClient, err := spade.NewClient(
		spade.InitBaseURL(url.URL{Scheme: "https", Host: "spade.twitch.tv", Path: "/"}),
	)
	if err != nil {
		return nil, err
	}

	var client HTTPClient
	if config.PROXY_ENABLED == "true" {
		client = proxyclient.New()
	} else {
		if config.DEBUG == "true" {
			client = &http.Client{
				Timeout: 10 * time.Second,
				Transport: &loghttp.Transport{
					LogRequest: func(req *http.Request) {
						fmt.Printf("[%p] %s %s", req, req.Method, req.URL)
					},
					LogResponse: func(resp *http.Response) {
						fmt.Printf("[%p] %d %s", resp.Request, resp.StatusCode, resp.Request.URL)
					},
				},
			}
		} else {
			client = &http.Client{Timeout: 10 * time.Second}
		}
	}

	return &BaseCrawler{
		platform:    platform,
		Log:         logrus.WithFields(logrus.Fields{"platform": platform}),
		SpadeClient: spadeClient,
		HTTPClient:  client,
	}, nil
}

func (bc *BaseCrawler) Platform() string {
	return bc.platform
}

func (bc *BaseCrawler) TrackChannelInfo(results <-chan model.ChannelInfo, expectedTotalViewers uint64) error {
	schemaName := "marionette_test"
	if os.Getenv("MARIONETTE_ENV") == "prod" {
		schemaName = "marionette_channel_prod"
	}
	totalChannels := 0
	totalViewer := int64(0)
	uniqueIDs := make(map[string]bool)
	var spadeEvents []spade.Event
	var firstEvent *spade.Event
	for channel := range results {
		_, exists := uniqueIDs[channel.ChannelID+channel.ChannelTitle]
		if exists {
			bc.Log.Warnf("conflict %s %d", channel.ChannelID, channel.Ccv)
		}
		uniqueIDs[channel.ChannelID+channel.ChannelTitle] = true
		totalChannels++
		totalViewer += channel.Ccv
		event := spade.Event{
			Name:       schemaName,
			Properties: channel,
		}
		spadeEvents = append(spadeEvents, event)

		if len(spadeEvents) >= batchSize {
			err := bc.SpadeClient.TrackEvents(
				context.Background(),
				spadeEvents...,
			)
			if err != nil {
				bc.Log.Errorf("Failed to track events %d: %s", len(spadeEvents), err.Error())
				spadeEvents = nil
				continue
			}
			spadeEvents = nil
		}
		if firstEvent == nil {
			firstEvent = &event
		}
	}
	if len(spadeEvents) != 0 {
		err := bc.SpadeClient.TrackEvents(
			context.Background(),
			spadeEvents...,
		)
		if err != nil {
			bc.Log.Errorf("Failed to track events %d: %s", len(spadeEvents), err.Error())
		}
	}

	if firstEvent == nil {
		bc.Log.Errorf("Crawl did not receive any channel")
		alert.SendAlert(alert.Alert{
			"Marionette Error",
			fmt.Sprintf("Failed to find channels %s", bc.Platform()),
		})
		cloudwatch.SendMetric("NoChannelFoundError", 1.0, cloudwatch.UnitCount, bc.platform)
	} else {
		bc.Log.Debugf(spew.Sdump(firstEvent))
	}

	expectedViewersString := *new(string)
	if expectedTotalViewers > 0 {
		expectedViewersString = fmt.Sprintf("(expected viewers: %d)", expectedTotalViewers)
	}

	bc.Log.Infof("total channels: %d viewers: %d %s", totalChannels, totalViewer, expectedViewersString)
	return nil
}
