package booyah

import (
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
	"net/url"
	"strconv"
	"strings"
	"time"

	crawler "code.justin.tv/esports-exp/marionette/pkg/crawler/base"
	"code.justin.tv/esports-exp/marionette/pkg/model"
	"code.justin.tv/esports-exp/marionette/pkg/util"
	"github.com/abadojack/whatlanggo"
)

const (
	sessionAPI = "https://booyah.live/api/v3/auths/sessions"
	gameAPI    = "https://booyah.live/api/v3/game-builds"
	streamAPI  = "https://booyah.live/api/v3/streams"
	deviceID   = "00657910-2064-11ea-a96e-63057d35ba50"
	pageLimit  = 100
	pageSize   = 50
)

type BooyahCrawler struct {
	crawler.BaseCrawler

	cookies string
	gameMap map[int]string
}

func NewCrawler() (*BooyahCrawler, error) {
	baseCrawler, err := crawler.NewBaseCrawler("booyah")
	if err != nil {
		return nil, err
	}

	return &BooyahCrawler{
		*baseCrawler,
		"",
		make(map[int]string),
	}, nil
}

type Stream struct {
	User struct {
		UID                    int    `json:"uid"`
		Nickname               string `json:"nickname"`
		NicknameNextUpdateTime int    `json:"nickname_next_update_time"`
		Thumbnail              string `json:"thumbnail"`
		Platform               int    `json:"platform"`
		FollowingCount         int    `json:"following_count"`
		FollowerCount          int    `json:"follower_count"`
	} `json:"user"`
	Channel struct {
		ChannelID          int    `json:"channel_id"`
		ChatroomID         int    `json:"chatroom_id"`
		Name               string `json:"name"`
		Description        string `json:"description"`
		OfflinePic         string `json:"offline_pic"`
		Thumbnail          string `json:"thumbnail"`
		CreateTime         int    `json:"create_time"`
		ShareURL           string `json:"share_url"`
		IsStreaming        bool   `json:"is_streaming"`
		IsVerifiedStreamer bool   `json:"is_verified_streamer"`
		IsEnableVod        bool   `json:"is_enable_vod"`
	} `json:"channel"`
	Stream struct {
		GameBuildID    int    `json:"game_build_id"`
		Snapshot       string `json:"snapshot"`
		StreamAddrList []struct {
			Resolution string `json:"resolution"`
			URLPath    string `json:"url_path"`
		} `json:"stream_addr_list"`
		MirrorList []struct {
			Name      string `json:"name"`
			URLDomain string `json:"url_domain"`
		} `json:"mirror_list"`
		DefaultMirror   string `json:"default_mirror"`
		ViewerCount     int    `json:"viewer_count"`
		ViewerCountInfo []struct {
			Platform  string `json:"platform"`
			LiveViews int    `json:"live_views"`
		} `json:"viewer_count_info"`
	} `json:"stream"`
}

type BooyahStreams struct {
	StreamList []Stream `json:"stream_list"`
	Cursor     int      `json:"cursor"`
	Version    int      `json:"version"`
}

type BooyahGames struct {
	GameBuildList []struct {
		GameBuildID    int    `json:"game_build_id"`
		GameID         int    `json:"game_id"`
		Name           string `json:"name"`
		TwitchName     string `json:"twitch_name"`
		Lang           string `json:"lang"`
		Platform       int    `json:"platform"`
		Picture        string `json:"picture"`
		Icon           string `json:"icon"`
		URLScheme      string `json:"url_scheme"`
		AppstoreID     string `json:"appstore_id"`
		PackageName    string `json:"package_name"`
		HighlightAvail int    `json:"highlight_avail"`
		FollowerCount  int    `json:"follower_count"`
		ViewerCount    int    `json:"viewer_count"`
	} `json:"game_build_list"`
	UpdateTs int `json:"update_ts"`
}

func (c *BooyahCrawler) getStreams(pageNum int) (*BooyahStreams, error) {
	req, err := http.NewRequest("GET", streamAPI, nil)
	if err != nil {
		return nil, fmt.Errorf("getLiveRoom failed to create request: %s", err.Error())
	}
	query := &url.Values{}
	query.Add("cursor", strconv.Itoa(pageNum*pageSize))
	query.Add("count", strconv.Itoa(pageSize))
	req.URL.RawQuery = query.Encode()

	req.Header.Add("authority", "booyah.live")
	req.Header.Add("accept", "application/json")
	req.Header.Add("user-agent", util.GetRandomUserAgent())
	req.Header.Add("origin", "https://booyah.live")
	req.Header.Add("sec-fetch-site", "same-origin")
	req.Header.Add("sec-fetch-mode", "cors")
	req.Header.Add("referer", "https://booyah.live/browse")
	req.Header.Add("Cookie", c.cookies)

	resp, err := util.ProxyReq(req)
	if err != nil {
		return nil, fmt.Errorf("getStreams failed to fetch: %s", err.Error())
	}
	defer resp.Body.Close()

	if resp.StatusCode != 200 {
		content, _ := ioutil.ReadAll(resp.Body)
		c.Log.Errorf("Error: %s", content)
		return nil, fmt.Errorf("Request failed: %d %s", resp.StatusCode, resp.Status)
	}

	result := BooyahStreams{}
	err = json.NewDecoder(resp.Body).Decode(&result)
	if err != nil {
		return nil, fmt.Errorf("getStreams failed to decode: %s", err.Error())
	}
	return &result, nil
}

func (c *BooyahCrawler) processStreams(streamList []Stream, timestamp int64, results chan<- model.ChannelInfo) {
	for _, stream := range streamList {
		var ccv int
		// Booyah no longer displays viewer count info list split by platform
		// for _, info := range stream.Stream.ViewerCountInfo {
		// 	if info.Platform == "mambettv" {
		// 		ccv = info.LiveViews
		// 	}
		// }
		ccv = stream.Stream.ViewerCount
		game, ok := c.gameMap[stream.Stream.GameBuildID]
		if !ok {
			game = "Other"
		}

		var lang string
		if langInfo := whatlanggo.Detect(stream.Channel.Name + "\n" + stream.Channel.Description); langInfo.Confidence > 0.5 {
			lang = langInfo.Lang.String()
		}

		results <- model.ChannelInfo{
			ChannelID:    strconv.Itoa(stream.Channel.ChannelID),
			ChannelName:  strings.TrimSpace(stream.User.Nickname),
			ChannelTitle: stream.Channel.Name,
			CountryCode:  "", // Unknown
			Game:         game,
			Language:     lang, // Heuristic
			Platform:     c.Platform(),
			Ccv:          int64(ccv),
			TimeCrawled:  timestamp,
		}
	}
}

func (c *BooyahCrawler) getSession() error {
	form := url.Values{
		"device_id": {deviceID},
		"language":  {"en"},
		"platform":  {"3"},
	}

	req, err := http.NewRequest("POST", sessionAPI, strings.NewReader(form.Encode()))

	if err != nil {
		return err
	}

	req.Header.Add("authority", "booyah.live")
	req.Header.Add("accept", "application/json")
	req.Header.Add("user-agent", util.GetRandomUserAgent())
	req.Header.Add("origin", "https://booyah.live")
	req.Header.Add("sec-fetch-site", "same-origin")
	req.Header.Add("sec-fetch-mode", "cors")
	req.Header.Add("referer", "https://booyah.live")

	resp, err := util.ProxyReq(req)
	if err != nil {
		return err
	}
	defer resp.Body.Close()
	if resp.StatusCode != 200 {
		return fmt.Errorf("booyah Request failed: %d %s", resp.StatusCode, resp.Status)
	}

	c.cookies = resp.Header.Get("Set-Cookie")
	c.Log.Infof("Got session: %s", c.cookies)

	return nil
}

func (c *BooyahCrawler) getGames() error {
	req, err := http.NewRequest("GET", gameAPI, nil)
	if err != nil {
		return err
	}
	query := &url.Values{}
	query.Add("platform", "0")
	query.Add("update_ts", "0")
	req.URL.RawQuery = query.Encode()

	req.Header.Add("authority", "booyah.live")
	req.Header.Add("accept", "application/json")
	req.Header.Add("user-agent", util.GetRandomUserAgent())
	req.Header.Add("origin", "https://booyah.live")
	req.Header.Add("sec-fetch-site", "same-origin")
	req.Header.Add("sec-fetch-mode", "cors")
	req.Header.Add("referer", "https://booyah.live")
	req.Header.Add("Cookie", c.cookies)

	resp, err := util.ProxyReq(req)
	if err != nil {
		return err
	}
	defer resp.Body.Close()
	if resp.StatusCode != 200 {
		return fmt.Errorf("booyah Request failed: %d %s", resp.StatusCode, resp.Status)
	}

	result := BooyahGames{}
	err = json.NewDecoder(resp.Body).Decode(&result)
	if err != nil {
		return err
	}

	for _, game := range result.GameBuildList {
		c.gameMap[game.GameBuildID] = game.Name
	}

	c.Log.Infof("Got games: %v", c.gameMap)
	return nil
}

func (c *BooyahCrawler) Crawl() error {
	timestamp := time.Now().Unix()
	err := c.getSession()
	if err != nil {
		return fmt.Errorf("Crawl failed to get session: %s", err.Error())
	}

	err = c.getGames()
	if err != nil {
		return fmt.Errorf("Crawl failed to get games: %s", err.Error())
	}

	results := make(chan model.ChannelInfo, pageLimit*pageSize)
	done := make(chan bool)

	go func() {
		c.TrackChannelInfo(results, 0)
		done <- true
	}()

	for i := 0; i < pageLimit; i++ {
		result, err := c.getStreams(i)
		if err != nil {
			c.Log.Errorf("Crawl failed to get live rooms %d: %s", i, err.Error())
			break
		}
		c.processStreams(result.StreamList, timestamp, results)

		if result.Cursor == 0 {
			c.Log.Infof("End of feed")
			break
		}
	}

	close(results)
	<-done

	return nil
}
