package crawler

import (
	"fmt"
	"time"

	log "github.com/sirupsen/logrus"

	"code.justin.tv/esports-exp/marionette/pkg/alert"
	"code.justin.tv/esports-exp/marionette/pkg/cloudwatch"
	"code.justin.tv/esports-exp/marionette/pkg/crawler/afreeca"
	"code.justin.tv/esports-exp/marionette/pkg/crawler/booyah"
	"code.justin.tv/esports-exp/marionette/pkg/crawler/cube"
	"code.justin.tv/esports-exp/marionette/pkg/crawler/dlive"
	"code.justin.tv/esports-exp/marionette/pkg/crawler/facebook"
	"code.justin.tv/esports-exp/marionette/pkg/crawler/mixer"
	"code.justin.tv/esports-exp/marionette/pkg/crawler/nimo"
	"code.justin.tv/esports-exp/marionette/pkg/crawler/youtube"
)

type Crawler interface {
	Crawl() error
	Platform() string
}

func MakeCrawler(platform string) (Crawler, error) {
	var crawler Crawler
	var err error

	switch platform {
	case "nimo":
		crawler, err = nimo.NewCrawler()
	case "cube":
		crawler, err = cube.NewCrawler()
	case "mixer":
		crawler, err = mixer.NewCrawler()
	case "dlive":
		crawler, err = dlive.NewCrawler()
	case "youtube":
		crawler, err = youtube.NewCrawler()
	case "facebook":
		crawler, err = facebook.NewCrawler()
	case "booyah":
		crawler, err = booyah.NewCrawler()
	case "afreeca":
		crawler, err = afreeca.NewCrawler()
	default:
		return nil, fmt.Errorf("Crawler for %s does not exist", platform)
	}

	if err != nil {
		return nil, err
	}

	return crawler, nil
}

func Run(c Crawler) {
	start := time.Now()

	if err := c.Crawl(); err != nil {
		log.Errorf("Crawling failed %s: %s", c.Platform(), err.Error())
		alert.SendAlert(alert.Alert{
			"Marionette Error",
			fmt.Sprintf("Failed to crawl %s: %s", c.Platform(), err.Error()),
		})
		cloudwatch.SendMetric("CrawlError", 1.0, cloudwatch.UnitCount, c.Platform())
	} else {
		log.Infof("Crawling done %s", c.Platform())
	}

	since := time.Since(start).Seconds()
	log.Infof("Crawling finished %s: %v seconds", c.Platform(), since)
}
