package cube

import (
	"encoding/json"
	"fmt"
	"net/http"
	"net/url"
	"os"
	"strconv"
	"strings"
	"sync"
	"time"

	crawler "code.justin.tv/esports-exp/marionette/pkg/crawler/base"
	"code.justin.tv/esports-exp/marionette/pkg/model"
	proxyclient "code.justin.tv/esports-exp/marionette/pkg/proxy/client"
	"code.justin.tv/esports-exp/marionette/pkg/util"
)

const (
	liveRoomAPI = "https://www.cube.tv/liveRoomApi/liveList"
	pageLimit   = 100
)

type CubeCrawler struct {
	crawler.BaseCrawler
}

func NewCrawler() (*CubeCrawler, error) {
	baseCrawler, err := crawler.NewBaseCrawler("cube")
	if err != nil {
		return nil, err
	}

	return &CubeCrawler{
		*baseCrawler,
	}, nil
}

type Room struct {
	Gid       string `json:"gid"`
	RoomType  string `json:"roomType"`
	UID       string `json:"uid"`
	UserCount string `json:"userCount"`
	Avatar    string `json:"avatar"`
	Country   string `json:"country"`
	Cover     string `json:"cover"`
	CubeID    string `json:"cube_id"`
	GameCover string `json:"gameCover"`
	GameID    string `json:"gameId"`
	GameTitle string `json:"gameTitle"`
	NickName  string `json:"nickName"`
	RoomTopic string `json:"roomTopic"`
	SnapShotl string `json:"snapShotl"`
	IsTheme   int    `json:"isTheme"`
}

type PaginatedData struct {
	List       []Room `json:"list"`
	TotalPage  int    `json:"totalPage"`
	TotalCount int    `json:"totalCount"`
	PageSize   int    `json:"pageSize"`
	Page       int    `json:"page"`
	ClientIP   string `json:"clientIp"`
}

type LivePageResult struct {
	Code int           `json:"code"`
	Msg  string        `json:"msg"`
	Data PaginatedData `json:"data"`
}

func (c *CubeCrawler) getLiveRoom(pageNum int) (*PaginatedData, error) {
	form := url.Values{
		"country": {"US"},
		"select":  {"ALL"},
		"gameId":  {""},
		"page":    {strconv.Itoa(pageNum)},
	}

	req, err := http.NewRequest("POST", liveRoomAPI, strings.NewReader(form.Encode()))
	if err != nil {
		return nil, fmt.Errorf("getLiveRoom failed to create request: %s", err.Error())
	}

	req.Header.Set("User-Agent", util.GetRandomUserAgent())
	req.Header.Set("Content-Type", "application/x-www-form-urlencoded")
	req.Header.Set("Accept", "application/json, text/plain, */*")
	req.Header.Set("Origin", "https://www.cube.tv")
	req.Header.Set("Referer", "https://www.cube.tv/g")

	var resp *http.Response
	if os.Getenv("PROXYSERVER_ENABLED") == "true" {
		client := proxyclient.New()
		resp, err = client.Do(req)
	} else {
		client := &http.Client{Timeout: 5 * time.Second}
		resp, err = client.Do(req)
	}
	if err != nil {
		return nil, fmt.Errorf("getLiveRoom failed to fetch: %s", err.Error())
	}
	defer resp.Body.Close()

	if resp.StatusCode != 200 {
		return nil, fmt.Errorf("cube Request failed: %d %s", resp.StatusCode, resp.Status)
	}

	result := LivePageResult{}
	err = json.NewDecoder(resp.Body).Decode(&result)
	if err != nil {
		return nil, fmt.Errorf("getLiveRoom failed to decode: %s", err.Error())
	}
	return &result.Data, nil
}

func (c *CubeCrawler) processLiveRoom(roomList []Room, timestamp int64, results chan<- model.ChannelInfo) {
	for _, room := range roomList {
		ccv, err := strconv.Atoi(room.UserCount)
		if err != nil {
			c.Log.Warningf("Failed to parse ccv: %s %v", room.UserCount, err)
			ccv = 0
		}
		results <- model.ChannelInfo{
			ChannelID:    room.CubeID,
			ChannelName:  room.NickName,
			ChannelTitle: room.RoomTopic,
			CountryCode:  room.Country,
			Game:         room.GameTitle,
			Language:     room.Country,
			Platform:     c.Platform(),
			Ccv:          int64(ccv),
			TimeCrawled:  timestamp,
		}
	}
}

func (c *CubeCrawler) getLiveRoomWithRetry(pageNum int, retryNum int, retryLimit int) (*PaginatedData, error) {
	result, err := c.getLiveRoom(pageNum)

	if err != nil {
		if retryNum < retryLimit {
			c.Log.Warnf("Crawl failed to make initial request page: %s, Retrying in 3s", err.Error())
			time.Sleep(2 * time.Second)
			return c.getLiveRoomWithRetry(pageNum, retryNum+1, retryLimit)
		}
		return nil, err
	}

	return result, nil
}

func (c *CubeCrawler) Crawl() error {
	timestamp := time.Now().Unix()

	result, err := c.getLiveRoomWithRetry(1, 0, 3)
	if err != nil {
		return fmt.Errorf("Crawl failed to make initial request page: %s", err.Error())
	}

	pageCount := result.TotalPage
	pageSize := result.PageSize

	results := make(chan model.ChannelInfo, pageCount*pageSize)
	done := make(chan bool)
	go func() {
		c.TrackChannelInfo(results, 0)
		done <- true
	}()

	c.processLiveRoom(result.List, timestamp, results)

	if pageCount > pageLimit {
		c.Log.Warnf("fetching too many pages: %d", pageCount)
		pageCount = pageLimit
	}

	wg := &sync.WaitGroup{}
	for i := 2; i <= pageCount; i++ {
		wg.Add(1)
		go func(pageIndex int) {
			defer wg.Done()
			result, err := c.getLiveRoom(pageIndex)
			if err != nil {
				c.Log.Errorf("Crawl failed to get live rooms %d: %s", pageIndex, err.Error())
				return
			}
			c.processLiveRoom(result.List, timestamp, results)
		}(i)
	}
	wg.Wait()

	close(results)
	<-done

	return nil
}
