package dlive

import (
	"encoding/json"
	"fmt"
	"net/http"
	"strings"
	"sync"
	"time"

	crawler "code.justin.tv/esports-exp/marionette/pkg/crawler/base"
	"code.justin.tv/esports-exp/marionette/pkg/model"
	"code.justin.tv/esports-exp/marionette/pkg/util"
)

const (
	graphqlAPI string = "https://graphigo.prd.dlive.tv/"
	pageSize   int    = 20
	batchSize  int    = 10
	pageLimit  int    = batchSize * 10
)

type DliveCrawler struct {
	crawler.BaseCrawler
}

func NewCrawler() (*DliveCrawler, error) {
	baseCrawler, err := crawler.NewBaseCrawler("dlive")
	if err != nil {
		return nil, err
	}

	return &DliveCrawler{
		*baseCrawler,
	}, nil
}

type Language struct {
	Language string `json:language`
}

type Category struct {
	Title string `json:"title"`
}

type Creator struct {
	ID          string `json:"id"`
	DisplayName string `json:"displayname"`
}

type Livestream struct {
	ID            string   `json:"id"`
	Title         string   `json:"title"`
	WatchingCount int      `json:"watchingCount"`
	Creator       Creator  `json:"creator"`
	Category      Category `json:"category"`
	Language      Language `json:"language"`
}

type LivestreamsPageResult struct {
	Data struct {
		Category struct {
			Livestreams struct {
				List []Livestream `json:"list"`
			} `json:"livestreams"`
		} `json:"category"`
	} `json:"data"`
}

func (c *DliveCrawler) getLiveRoom(pageNum int, pageSize int) (*LivestreamsPageResult, error) {
	query := fmt.Sprintf(`{
		"operationName": "CategoryLivestreamsPage",
		"variables": {
			"id": "0",
			"opt": {
				"first": %d,
				"after": "%d",
				"languageID": null,
				"showNSFW": true,
				"order": "TRENDING"
			}
		},
		"extensions": {
			"persistedQuery": {
				"version": 1,
				"sha256Hash": "6ae0d1bf97d81c2f3b383e30662f934f3d060f35335ab5819e86ae61151b72ed"
			}
		}
	}`, pageSize, pageNum*pageSize-1)
	req, err := http.NewRequest("POST", graphqlAPI, strings.NewReader(query))

	if err != nil {
		return nil, fmt.Errorf("getLiveRoom failed to create request: %s", err.Error())
	}

	req.Header.Set("User-Agent", util.GetRandomUserAgent())
	req.Header.Add("Sec-Fetch-Mode", "cors")
	req.Header.Add("x-dlive-mversion", "v0.6.64")
	req.Header.Add("accept", "*/*")
	req.Header.Add("x-dlive-mtype", "web")
	req.Header.Add("Referer", "https://dlive.tv/s/browse/0/All")
	req.Header.Add("content-type", "application/json")
	req.Header.Add("Cache-Control", "no-cache")
	req.Header.Add("Accept-Encoding", "gzip, deflate")

	resp, err := c.HTTPClient.Do(req)
	if err != nil {
		return nil, fmt.Errorf("getLiveRoom failed to fetch: %s", err.Error())
	}
	defer resp.Body.Close()

	if resp.StatusCode != 200 {
		return nil, fmt.Errorf("Request failed: %d %s", resp.StatusCode, resp.Status)
	}

	respJSON := LivestreamsPageResult{}
	err = json.NewDecoder(resp.Body).Decode(&respJSON)
	if err != nil {
		return nil, fmt.Errorf("getLiveRoom failed to decode: %s", err.Error())
	}
	return &respJSON, nil
}

func (c *DliveCrawler) processLiveRoom(livestreams *LivestreamsPageResult, timestamp int64, results chan<- model.ChannelInfo) {
	for _, room := range livestreams.Data.Category.Livestreams.List {
		results <- model.ChannelInfo{
			ChannelID:    room.ID,
			ChannelName:  room.Creator.DisplayName,
			ChannelTitle: room.Title,
			CountryCode:  room.Language.Language,
			Game:         room.Category.Title,
			Language:     room.Language.Language,
			Platform:     c.Platform(),
			Ccv:          int64(room.WatchingCount),
			TimeCrawled:  timestamp,
		}
	}
}

func (c *DliveCrawler) Crawl() error {
	timestamp := time.Now().Unix()
	results := make(chan model.ChannelInfo, pageLimit*pageSize)
	done := make(chan bool)
	go func() {
		c.TrackChannelInfo(results, 0)
		done <- true
	}()

	allDone := false
	for i := 0; i < int(pageLimit/batchSize) && !allDone; i++ {
		wg := &sync.WaitGroup{}
		for j := 0; j < batchSize && !allDone; j++ {
			wg.Add(1)
			go func(pageIndex int) {
				defer wg.Done()
				result, err := c.getLiveRoom(pageIndex, pageSize)
				if err != nil {
					c.Log.Errorf("Crawl failed to get live rooms %d: %s", pageIndex, err.Error())
					return
				}
				if len(result.Data.Category.Livestreams.List) == 0 {
					allDone = true
				}
				c.processLiveRoom(result, timestamp, results)
			}(i*batchSize + j)
		}
		wg.Wait()
	}
	if !allDone {
		c.Log.Warnf("fetching too many pages")
	}
	close(results)
	<-done

	return nil
}
