package facebook

import (
	"encoding/json"
	"fmt"
	"net/http"
	"net/url"
	"os"
	"strings"
	"sync"
	"time"

	"github.com/abadojack/whatlanggo"

	crawler "code.justin.tv/esports-exp/marionette/pkg/crawler/base"
	"code.justin.tv/esports-exp/marionette/pkg/model"
	proxyclient "code.justin.tv/esports-exp/marionette/pkg/proxy/client"
	"code.justin.tv/esports-exp/marionette/pkg/util"
)

const (
	graphqlAPI string = "https://www.facebook.com/api/graphql/"
)

type FacebookCrawler struct {
	crawler.BaseCrawler
}

func NewCrawler() (*FacebookCrawler, error) {
	baseCrawler, err := crawler.NewBaseCrawler("facebook")
	if err != nil {
		return nil, err
	}

	return &FacebookCrawler{
		*baseCrawler,
	}, nil
}

type Media struct {
	URL  string `json:"url"`
	Game struct {
		Name string `json:"name"`
	} `json:"attributed_game"`
	Owner struct {
		ID   string `json:"id"`
		Name string `json:"name"`
	} `json:"owner"`
	BroadcastStatus    string `json:"broadcast_status"`
	ViewerCount        int    `json:"viewer_count"`
	Name               string `json:"name"`
	SavableDescription struct {
		Text string `json:"text"`
	} `json:"savable_description"`
}

type Node struct {
	Node struct {
		FeedUnit struct {
			Attachments []struct {
				Media Media `json:"media"`
			} `json:"attachments`
		} `json:"feed_unit"`
	} `json:"node"`
}

type PageInfo struct {
	HasNextPage bool   `json:"has_next_page"`
	EndCursor   string `json:"end_cursor"`
}

type QueryResult struct {
	Data struct {
		Node struct {
			SectionComponents struct {
				PageInfo PageInfo `json:"page_info"`
				Edges    []Node   `json:"edges"`
			} `json:"section_components"`
		} `json:"node"`
	} `json:"data"`
}

type DirectoryNode struct {
	Node struct {
		ID   string `json:"id"`
		Name string `json:"name"`
		CVC  int    `json:"cvc"`
	} `json:"node"`
}

type DirectoryResult struct {
	Data struct {
		GameQuery []struct {
			Results struct {
				PageInfo PageInfo        `json:"page_info"`
				Edges    []DirectoryNode `json:"edges"`
			} `json:"results"`
		} `json:"game_query"`
	} `json:"data"`
}

type SectionNode struct {
	Node struct {
		ID   string `json:"id"`
		Type string `json:"section_type_name"`
	}
}

type SectionResult struct {
	Data struct {
		GamesVideoHome struct {
			GamesVideoSections struct {
				Edges []SectionNode `json:"edges"`
			} `json:"games_video_sections"`
		} `json:"games_video_home"`
	} `json:"data"`
}

func (c *FacebookCrawler) apiCall(req *http.Request) (*http.Response, error) {
	req.Header.Set("User-Agent", util.GetRandomUserAgent())
	req.Header.Add("sec-fetch-mode", "cors")
	req.Header.Add("origin", "https://www.facebook.com")
	req.Header.Add("accept-language", "en-US")
	req.Header.Add("content-type", "application/x-www-form-urlencoded")
	req.Header.Add("accept", "application/json")
	req.Header.Add("referer", "https://www.facebook.com/gaming/?section_id=dmg6MTg5MTU5MTMzMDg4MTE5Ng%3D%3D&view=all&previous_view=home")
	req.Header.Add("authority", "www.facebook.com")
	req.Header.Add("sec-fetch-site", "same-origin")
	req.Header.Add("cache-control", "no-cache")

	var err error
	var resp *http.Response
	if os.Getenv("PROXYSERVER_ENABLED") == "true" {
		client := proxyclient.New()
		resp, err = client.Do(req)
	} else {
		client := &http.Client{Timeout: 5 * time.Second}
		resp, err = client.Do(req)
	}

	if err != nil {
		return nil, fmt.Errorf("apiCall failed to fetch: %s", err.Error())
	}

	if resp.StatusCode != 200 {
		return nil, fmt.Errorf("Request failed: %d %s", resp.StatusCode, resp.Status)
	}

	return resp, nil
}

func (c *FacebookCrawler) getFirstPage(sectionID string) (*QueryResult, error) {
	// facebook doesn't really honor the pageSize param ¯\_(ツ)_/¯
	form := url.Values{
		"av":                       {"0"},
		"__user":                   {"0"},
		"__a":                      {"1"},
		"__req":                    {"q"},
		"__be":                     {"1"},
		"__pc":                     {"PHASED:DEFAULT"},
		"dpr":                      {"1"},
		"__spin_b":                 {"trunk"},
		"fb_api_caller_class":      {"RelayModern"},
		"fb_api_req_friendly_name": {"GamingVideoSeeAllListSectionQuery"},
		"variables":                {fmt.Sprintf(`{"count":4,"sectionID":"%s", "caller":"video_home_channel_see_all"}`, sectionID)},
		"doc_id":                   {"1766919656745913"},
	}

	req, err := http.NewRequest("POST", graphqlAPI, strings.NewReader(form.Encode()))

	if err != nil {
		return nil, fmt.Errorf("getFirstPage failed to create request: %s", err.Error())
	}

	resp, err := c.apiCall(req)
	if err != nil {
		return nil, fmt.Errorf("Api call failed: %s", err.Error())
	}
	defer resp.Body.Close()

	respJSON := QueryResult{}
	err = json.NewDecoder(resp.Body).Decode(&respJSON)
	if err != nil {
		return nil, fmt.Errorf("getFirstPage failed to decode: %s", err.Error())
	}

	return &respJSON, nil
}

func (c *FacebookCrawler) getNextPage(sectionID string, cursor string) (*QueryResult, error) {
	form := url.Values{
		"av":                       {"0"},
		"__user":                   {"0"},
		"__a":                      {"1"},
		"__req":                    {"q"},
		"__be":                     {"1"},
		"__pc":                     {"PHASED:DEFAULT"},
		"dpr":                      {"1"},
		"__spin_b":                 {"trunk"},
		"fb_api_caller_class":      {"RelayModern"},
		"fb_api_req_friendly_name": {"GamingVideoVideosListPaginationQuery"},
		"variables":                {fmt.Sprintf(`{"count":4,"cursor":"%s","sectionID":"%s"}`, cursor, sectionID)},
		"doc_id":                   {"2476138682402457"},
	}

	req, err := http.NewRequest("POST", graphqlAPI, strings.NewReader(form.Encode()))

	if err != nil {
		return nil, fmt.Errorf("getNextPage failed to create request: %s", err.Error())
	}

	resp, err := c.apiCall(req)
	if err != nil {
		return nil, fmt.Errorf("Api call failed: %s", err.Error())
	}

	respJSON := QueryResult{}
	err = json.NewDecoder(resp.Body).Decode(&respJSON)
	if err != nil {
		return nil, fmt.Errorf("getNextPage failed to decode: %s", err.Error())
	}

	return &respJSON, nil
}

func (c *FacebookCrawler) getFirstDirectory() (*DirectoryResult, error) {
	// facebook doesn't really honor the count param ¯\_(ツ)_/¯
	form := url.Values{
		"av":                       {"0"},
		"__user":                   {"0"},
		"__a":                      {"1"},
		"__req":                    {"q"},
		"__be":                     {"1"},
		"__pc":                     {"PHASED:DEFAULT"},
		"dpr":                      {"1"},
		"__spin_b":                 {"trunk"},
		"fb_api_caller_class":      {"RelayModern"},
		"fb_api_req_friendly_name": {"GamesVideoDirectoryContentQuery"},
		"variables":                {`{"params":[{"fb_source":2416,"game_query_type":"TOP_GAMES"}],"count":16}`},
		"doc_id":                   {"2822483281100058"},
	}

	req, err := http.NewRequest("POST", graphqlAPI, strings.NewReader(form.Encode()))

	if err != nil {
		return nil, fmt.Errorf("getFirstDirectory failed to create request: %s", err.Error())
	}

	resp, err := c.apiCall(req)
	if err != nil {
		return nil, fmt.Errorf("getFirstDirectory Api call failed: %s", err.Error())
	}
	defer resp.Body.Close()

	respJSON := DirectoryResult{}
	err = json.NewDecoder(resp.Body).Decode(&respJSON)
	if err != nil {
		return nil, fmt.Errorf("getFirstDirectory failed to decode: %s", err.Error())
	}

	return &respJSON, nil
}

func (c *FacebookCrawler) getNextDirectory(cursor string) (*DirectoryResult, error) {
	// facebook doesn't really honor the count param ¯\_(ツ)_/¯
	form := url.Values{
		"av":                       {"0"},
		"__user":                   {"0"},
		"__a":                      {"1"},
		"__req":                    {"q"},
		"__be":                     {"1"},
		"__pc":                     {"PHASED:DEFAULT"},
		"dpr":                      {"1"},
		"__spin_b":                 {"trunk"},
		"fb_api_caller_class":      {"RelayModern"},
		"fb_api_req_friendly_name": {"GamesVideoDirectoryGameListPaginationQuery"},
		"variables":                {fmt.Sprintf(`{"params":[{"fb_source":2416,"game_query_type":"TOP_GAMES"}],"count":4,"cursor":"%s"}`, cursor)},
		"doc_id":                   {"2555379191147181"},
	}

	req, err := http.NewRequest("POST", graphqlAPI, strings.NewReader(form.Encode()))

	if err != nil {
		return nil, fmt.Errorf("getNextDirectory failed to create request: %s", err.Error())
	}

	resp, err := c.apiCall(req)
	if err != nil {
		return nil, fmt.Errorf("getNextDirectory Api call failed: %s", err.Error())
	}
	defer resp.Body.Close()

	respJSON := DirectoryResult{}
	err = json.NewDecoder(resp.Body).Decode(&respJSON)
	if err != nil {
		return nil, fmt.Errorf("getNextDirectory failed to decode: %s", err.Error())
	}

	return &respJSON, nil
}

func (c *FacebookCrawler) getLiveSectionID(node *DirectoryNode) (*string, error) {
	// facebook doesn't really honor the count param ¯\_(ツ)_/¯
	form := url.Values{
		"av":                       {"0"},
		"__user":                   {"0"},
		"__a":                      {"1"},
		"__req":                    {"q"},
		"__be":                     {"1"},
		"__pc":                     {"PHASED:DEFAULT"},
		"dpr":                      {"1"},
		"__spin_b":                 {"trunk"},
		"fb_api_caller_class":      {"RelayModern"},
		"fb_api_req_friendly_name": {"GamesVideoSingleGameVideoQuery"},
		"variables":                {fmt.Sprintf(`{"params":{"game_id":"%s"},"count":4}`, node.Node.ID)},
		"doc_id":                   {"2572454659471348"},
	}

	req, err := http.NewRequest("POST", graphqlAPI, strings.NewReader(form.Encode()))

	if err != nil {
		return nil, fmt.Errorf("getLiveSectionID failed to create request: %s", err.Error())
	}

	resp, err := c.apiCall(req)
	if err != nil {
		return nil, fmt.Errorf("getLiveSectionID Api call failed: %s", err.Error())
	}
	defer resp.Body.Close()

	respJSON := SectionResult{}
	err = json.NewDecoder(resp.Body).Decode(&respJSON)
	if err != nil {
		return nil, fmt.Errorf("getLiveSectionID failed to decode: %s", err.Error())
	}

	for _, node := range respJSON.Data.GamesVideoHome.GamesVideoSections.Edges {
		if node.Node.Type == "GAMES_VIDEO_SINGLE_GAME_LIVE_NOW" {
			return &node.Node.ID, nil
		}
	}

	return nil, fmt.Errorf("Cannot find live section")
}

func (c *FacebookCrawler) processLiveRoom(livestreams []Node, timestamp int64, results chan<- model.ChannelInfo) {
	for _, room := range livestreams {
		if len(room.Node.FeedUnit.Attachments) > 0 {
			var lang string
			if langInfo := whatlanggo.Detect(room.Node.FeedUnit.Attachments[0].Media.SavableDescription.Text); langInfo.Confidence > 0.5 {
				lang = langInfo.Lang.String()
			}

			results <- model.ChannelInfo{
				ChannelID:    room.Node.FeedUnit.Attachments[0].Media.Owner.ID,
				ChannelName:  room.Node.FeedUnit.Attachments[0].Media.Owner.Name,
				ChannelTitle: room.Node.FeedUnit.Attachments[0].Media.Name,
				CountryCode:  "", // TODO
				Game:         room.Node.FeedUnit.Attachments[0].Media.Game.Name,
				Language:     lang,
				Platform:     c.Platform(),
				Ccv:          int64(room.Node.FeedUnit.Attachments[0].Media.ViewerCount),
				TimeCrawled:  timestamp,
			}
		} else {
			c.Log.Warnf("no attachment for node")
		}
	}
}

func (c *FacebookCrawler) Crawl() error {
	timestamp := time.Now().Unix()

	var games []DirectoryNode
	totalViewers := uint64(0)
	dr, err := c.getFirstDirectory()
	for {
		if err != nil {
			c.Log.Errorf("rip query: %s", err.Error())
			break
		}

		if len(dr.Data.GameQuery) == 0 {
			c.Log.Errorf("didn't load any directories")
			break
		}

		c.Log.Debugf("loaded %d directories", len(dr.Data.GameQuery[0].Results.Edges))

		for _, node := range dr.Data.GameQuery[0].Results.Edges {
			totalViewers += uint64(node.Node.CVC)
			if node.Node.CVC > 0 {
				games = append(games, node)
			}
		}

		// fetch next page
		if !dr.Data.GameQuery[0].Results.PageInfo.HasNextPage {
			break
		}
		dr, err = c.getNextDirectory(dr.Data.GameQuery[0].Results.PageInfo.EndCursor)
	}

	c.Log.Infof("loaded %d game categories with %d viewers", len(games), totalViewers)

	results := make(chan model.ChannelInfo, len(games)*100)
	done := make(chan bool)
	go func() {
		c.TrackChannelInfo(results, totalViewers)
		done <- true
	}()

	wg := &sync.WaitGroup{}
	for _, node := range games {
		wg.Add(1)
		go func(node DirectoryNode) {
			defer wg.Done()
			sectionID, err := c.getLiveSectionID(&node)
			if err != nil {
				c.Log.Errorf("no live section %s: %s", node.Node.Name, err.Error())
				return
			}
			result, err := c.getFirstPage(*sectionID)
			if err == nil {
				c.Log.Debugf("First page %s: %d", node.Node.Name, len(result.Data.Node.SectionComponents.Edges))
			}

			pageCnt, liveRoomCnt := 1, 0
			for {
				if err != nil {
					c.Log.Errorf("rip query %s: %s", node.Node.Name, err.Error())
					break
				}
				if len(result.Data.Node.SectionComponents.Edges) > 0 {
					c.processLiveRoom(result.Data.Node.SectionComponents.Edges, timestamp, results)
					liveRoomCnt += len(result.Data.Node.SectionComponents.Edges)
				} else {
					c.Log.Warnf("No live room for [%s] on page[%d]: %v", node.Node.Name, pageCnt, result)
				}

				// fetch next page
				if !result.Data.Node.SectionComponents.PageInfo.HasNextPage {
					break
				}
				result, err = c.getNextPage(*sectionID, result.Data.Node.SectionComponents.PageInfo.EndCursor)
				pageCnt++
				time.Sleep(100 * time.Millisecond)
			}
			c.Log.Infof("crawled [%d] page for [%s] which has live room: [%d]", pageCnt, node.Node.Name, liveRoomCnt)
		}(node)
		time.Sleep(100 * time.Millisecond)
	}
	wg.Wait()

	close(results)
	<-done

	return nil
}
