package mixer

import (
	"encoding/json"
	"fmt"
	"net/http"
	"net/url"
	"os"
	"strconv"
	"sync"
	"time"

	crawler "code.justin.tv/esports-exp/marionette/pkg/crawler/base"
	"code.justin.tv/esports-exp/marionette/pkg/model"
	proxyclient "code.justin.tv/esports-exp/marionette/pkg/proxy/client"
	"code.justin.tv/esports-exp/marionette/pkg/util"
)

const (
	channelsApi string = "https://mixer.com/api/v1/channels"
	pageSize    int    = 100
	batchSize   int    = 32
	pageLimit   int    = batchSize * 4
)

type MixerCrawler struct {
	crawler.BaseCrawler
}

func NewCrawler() (*MixerCrawler, error) {
	baseCrawler, err := crawler.NewBaseCrawler("mixer")
	if err != nil {
		return nil, err
	}

	return &MixerCrawler{
		*baseCrawler,
	}, nil
}

type User struct {
	Username string `json:"username"`
}

type Type struct {
	Name string `json:"name"`
}

type Channel struct {
	ID            int    `json:"id"`
	Name          string `json:"name"`
	ViewersCurret int    `json:"viewersCurrent"`
	LanguageID    string `json:"languageId"`
	User          User   `json:"user"`
	Type          Type   `json:"type"`
}

type ChannelResult = []Channel

func (c *MixerCrawler) getLiveRoom(pageNum int, pageSize int) (*ChannelResult, error) {
	req, err := http.NewRequest("GET", channelsApi, nil)
	q := url.Values{}
	q.Add("order", "viewersCurrent:DESC")
	q.Add("where", "viewersCurrent:ne:0")
	q.Add("limit", strconv.Itoa(pageSize))
	q.Add("page", strconv.Itoa(pageNum))
	req.URL.RawQuery = q.Encode()

	if err != nil {
		return nil, fmt.Errorf("getLiveRoom failed to create request: %s", err.Error())
	}

	req.Header.Set("User-Agent", util.GetRandomUserAgent())
	req.Header.Set("Referer", "https://mixer.com")

	var resp *http.Response
	if os.Getenv("PROXYSERVER_ENABLED") == "true" {
		client := proxyclient.New()
		resp, err = client.Do(req)
	} else {
		client := &http.Client{Timeout: 5 * time.Second}
		resp, err = client.Do(req)
	}

	if err != nil {
		return nil, fmt.Errorf("getLiveRoom failed to fetch: %s", err.Error())
	}
	defer resp.Body.Close()

	if resp.StatusCode != 200 {
		return nil, fmt.Errorf("Request failed: %d %s", resp.StatusCode, resp.Status)
	}

	respJSON := ChannelResult{}
	err = json.NewDecoder(resp.Body).Decode(&respJSON)
	if err != nil {
		return nil, fmt.Errorf("getLiveRoom failed to decode: %s", err.Error())
	}
	return &respJSON, nil
}

func (c *MixerCrawler) processLiveRoom(channels *ChannelResult, timestamp int64, results chan<- model.ChannelInfo) {
	for _, room := range *channels {
		results <- model.ChannelInfo{
			ChannelID:    strconv.Itoa(room.ID),
			ChannelName:  room.User.Username,
			ChannelTitle: room.Name,
			CountryCode:  room.LanguageID,
			Game:         room.Type.Name,
			Language:     room.LanguageID,
			Platform:     c.Platform(),
			Ccv:          int64(room.ViewersCurret),
			TimeCrawled:  timestamp,
		}
	}
}

func (c *MixerCrawler) Crawl() error {
	timestamp := time.Now().Unix()
	results := make(chan model.ChannelInfo, pageLimit*pageSize)
	done := make(chan bool)
	go func() {
		c.TrackChannelInfo(results, 0)
		done <- true
	}()

	allDone := false
	for i := 0; i < int(pageLimit/batchSize) && !allDone; i++ {
		wg := &sync.WaitGroup{}
		for j := 0; j < batchSize && !allDone; j++ {
			wg.Add(1)
			go func(pageIndex int) {
				defer wg.Done()
				result, err := c.getLiveRoom(pageIndex, pageSize)
				if err != nil {
					c.Log.Errorf("Crawl failed to get live rooms %d: %s", pageIndex, err.Error())
					return
				}
				if len(*result) == 0 {
					allDone = true
				}
				c.processLiveRoom(result, timestamp, results)
			}(i*batchSize + j)
		}
		wg.Wait()
	}
	if !allDone {
		c.Log.Warnf("fetching too many pages")
	}

	close(results)
	<-done

	return nil
}
