package nimo

import (
	"encoding/json"
	"fmt"
	"net/http"
	"net/url"
	"os"
	"strconv"
	"strings"
	"sync"
	"time"

	crawler "code.justin.tv/esports-exp/marionette/pkg/crawler/base"
	"code.justin.tv/esports-exp/marionette/pkg/model"
	proxyclient "code.justin.tv/esports-exp/marionette/pkg/proxy/client"
	"code.justin.tv/esports-exp/marionette/pkg/util"
)

const (
	// pageNum, pageSize
	liveRoomAPI string = "https://api.nimo.tv/oversea/nimo/api/v2/liveRoom/liveRoomPage-%d-%d-/US/1033/1000"
	pageSize    int    = 100
	pageLimit   int    = 100
)

type NimoCrawler struct {
	crawler.BaseCrawler
}

func NewCrawler() (*NimoCrawler, error) {
	baseCrawler, err := crawler.NewBaseCrawler("nimo")
	if err != nil {
		return nil, err
	}

	return &NimoCrawler{
		*baseCrawler,
	}, nil
}

type GameEntity struct {
	ClientID int    `json:"clientId"`
	GameID   int    `json:"gameId"`
	GameName string `json:"gameName"`
	ID       int    `json:"id"`
	Logo     string `json:"logo"`
	Name     string `json:"name"`
	NameLang string `json:"nameLang"`
	Platform int    `json:"platform"`
	Pos      int    `json:"pos"`
}

type LiveRoom struct {
	Alise               string `json:"alise"`
	AnchorCountryCode   string `json:"anchorCountryCode"`
	AnchorID            int    `json:"anchorId"`
	AnchorName          string `json:"anchorName"`
	AuthenticatedAnchor int    `json:"authenticatedAnchor"`
	BusinessType        int    `json:"businessType"`
	CreatedTime         int64  `json:"createdTime"`
	EndLiveTime         int64  `json:"endLiveTime"`
	ID                  int    `json:"id"`

	LiveStreamStatus int    `json:"liveStreamStatus"`
	IsPlayback       int    `json:"isPlayback"`
	MicroPKStatus    int    `json:"microPKStatus"`
	OnlineStatus     int    `json:"onlineStatus"`
	Lcid             int    `json:"lcid"`
	LcidText         string `json:"lcidText"`
	RoomTheme        string `json:"roomTheme"`
	RoomType         int    `json:"roomType"`
	RoomTypeName     string `json:"roomTypeName"`

	UpdatedTime int64 `json:"updatedTime"`
	ViewerNum   int   `json:"viewerNum"`
}

type PageControl struct {
	Count     int `json:"count"`
	PageCount int `json:"pageCount"`
	PageIndex int `json:"pageIndex"`
	PageSize  int `json:"pageSize"`
	PageStep  int `json:"pageStep"`
}

type LiveRoomViewResult struct {
	LCID             int          `json:"lCID"`
	GameEntityList   []GameEntity `json:"gameEntityList"`
	LiveRoomViewList []LiveRoom   `json:"liveRoomViewList"`
	PageControlView  PageControl  `json:"pageControlView"`
}

type LiveRoomViewResponse struct {
	Code    int    `json:"code"`
	Message string `json:"message"`
	Data    struct {
		KeyType int                `json:"keyType"`
		Result  LiveRoomViewResult `json:"result"`
	} `json:"data"`
}

func (c *NimoCrawler) getLiveRoom(pageNum int, pageSize int) (*LiveRoomViewResult, error) {
	apiURL := fmt.Sprintf(liveRoomAPI, pageNum, pageSize)

	form := url.Values{
		"keyType": {"0"},
		"body":    {"{\"requestSource\":\"WEB\"}"},
	}

	req, err := http.NewRequest("POST", apiURL, strings.NewReader(form.Encode()))
	if err != nil {
		return nil, fmt.Errorf("getLiveRoom failed to create request: %s", err.Error())
	}

	req.Header.Set("User-Agent", util.GetRandomUserAgent())
	req.Header.Set("Content-Type", "application/x-www-form-urlencoded")
	req.Header.Set("Origin", "https://www.nimo.tv")
	req.Header.Set("Referer", "https://www.nimo.tv/lives")

	var resp *http.Response
	if os.Getenv("PROXYSERVER_ENABLED") == "true" {
		client := proxyclient.New()
		resp, err = client.Do(req)
	} else {
		client := &http.Client{Timeout: 5 * time.Second}
		resp, err = client.Do(req)
	}

	if err != nil {
		return nil, fmt.Errorf("getLiveRoom failed to fetch: %s", err.Error())
	}
	defer resp.Body.Close()

	if resp.StatusCode != 200 {
		return nil, fmt.Errorf("Request failed: %d %s", resp.StatusCode, resp.Status)
	}

	respJSON := LiveRoomViewResponse{}
	err = json.NewDecoder(resp.Body).Decode(&respJSON)
	if err != nil {
		return nil, fmt.Errorf("getLiveRoom failed to decode: %s", err.Error())
	}
	return &respJSON.Data.Result, nil
}

func (c *NimoCrawler) processLiveRoom(liveRoomViewList []LiveRoom, timestamp int64, results chan<- model.ChannelInfo) {
	for _, room := range liveRoomViewList {
		results <- model.ChannelInfo{
			ChannelID:    strconv.Itoa(room.ID),
			ChannelName:  room.AnchorName,
			ChannelTitle: room.RoomTheme,
			CountryCode:  room.AnchorCountryCode,
			Game:         room.RoomTypeName,
			Language:     room.LcidText,
			Platform:     c.Platform(),
			Ccv:          int64(room.ViewerNum),
			TimeCrawled:  timestamp,
		}
	}
}

func (c *NimoCrawler) Crawl() error {
	timestamp := time.Now().Unix()
	result, err := c.getLiveRoom(1, pageSize)
	if err != nil {
		return fmt.Errorf("Crawl failed to make initial request page: %s", err.Error())
	}

	pageCount := result.PageControlView.PageCount

	results := make(chan model.ChannelInfo, pageCount*pageSize)
	done := make(chan bool)
	go func() {
		c.TrackChannelInfo(results, 0)
		done <- true
	}()

	c.processLiveRoom(result.LiveRoomViewList, timestamp, results)

	if pageCount > pageLimit {
		c.Log.Warnf("fetching too many pages %d", pageCount)
		pageCount = pageLimit
	}

	wg := &sync.WaitGroup{}
	for i := 2; i <= pageCount; i++ {
		wg.Add(1)
		go func(pageIndex int) {
			defer wg.Done()
			result, err := c.getLiveRoom(pageIndex, pageSize)
			if err != nil {
				c.Log.Errorf("Crawl failed to get live rooms %d", pageIndex)
				return
			}
			c.processLiveRoom(result.LiveRoomViewList, timestamp, results)
		}(i)
	}
	wg.Wait()

	close(results)
	<-done

	return nil
}
