package youtube

import (
	"bytes"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"net/http"
	"net/url"
	"strings"
	"sync"
	"time"

	"github.com/abadojack/whatlanggo"
	"github.com/dustin/go-humanize"

	crawler "code.justin.tv/esports-exp/marionette/pkg/crawler/base"
	"code.justin.tv/esports-exp/marionette/pkg/model"
	"code.justin.tv/esports-exp/marionette/pkg/util"
)

const (
	// pageNum, pageSize
	gameAPI    string = "https://m.youtube.com/gaming/games"
	gameAPIWeb string = "https://www.youtube.com/gaming/games"
	liveAPI    string = "https://m.youtube.com/channel/%s/live"
	liveWebAPI string = "https://www.youtube.com/channel/%s/live"
	ajaxAPI    string = "https://www.youtube.com/browse_ajax"
	youtubeAPI string = "https://www.youtube.com/youtubei/v1/browse?key=AIzaSyAO_FJ2SlqU8Q4STEHLGCilw_Y9_11qcW8"

	//INNERTUBE_API_KEY string = "AIzaSyAO_FJ2SlqU8Q4STEHLGCilw_Y9_11qcW8"

	// stream view counts state.
	Watching = "watching"
	Views    = "views"
)

var (
	visitorData = ""
)

type YoutubeCrawler struct {
	crawler.BaseCrawler
	cookies string
}

func NewCrawler() (*YoutubeCrawler, error) {
	baseCrawler, err := crawler.NewBaseCrawler("youtube")
	if err != nil {
		return nil, err
	}

	return &YoutubeCrawler{
		*baseCrawler,
		"",
	}, nil
}

type GameCard struct {
	Game struct {
		GameDetailsRenderer struct {
			Title struct {
				Runs []struct {
					Text string `json:"text"`
				} `json:"runs"`
				SimpleText string `json:"simpleText"`
			} `json:"title"`
			BoxArt struct {
				Thumbnails []struct {
					URL string `json:"url"`
				} `json:"thumbnails"`
			} `json:"boxArt"`
			BoxArtOverlayText struct {
				Runs []struct {
					Text string `json:"text"`
				} `json:"runs"`
			} `json:"boxArtOverlayText"`
			Endpoint struct {
				ClickTrackingParams string `json:"clickTrackingParams"`
				CommandMetadata     struct {
					WebCommandMetadata struct {
						URL string `json:"url"`
					} `json:"webCommandMetadata"`
				} `json:"commandMetadata"`
				BrowseEndpoint struct {
					BrowseID string `json:"browseId"`
				} `json:"browseEndpoint"`
			} `json:"endpoint"`
			ReleaseDate struct {
				Runs []struct {
					Text string `json:"text"`
				} `json:"runs"`
			} `json:"releaseDate"`
			TrackingParams  string `json:"trackingParams"`
			LiveViewersText struct {
				Runs []struct {
					Text string `json:"text"`
				} `json:"runs"`
			} `json:"liveViewersText"`
			IsOfficialBoxArt bool `json:"isOfficialBoxArt"`
		} `json:"gameDetailsRenderer"`
	} `json:"game"`
	TrackingParams string `json:"trackingParams"`
}

type GameAndContinuationItem struct {
	GameCardRenderer         GameCard          `json:"gameCardRenderer,omitempty"`
	ContinuationItemRenderer *ContinuationItem `json:"continuationItemRenderer,omitempty"`
}

type ContinuationItem struct {
	ContinuationEndpoint struct {
		ClickTrackingParams string `json:"clickTrackingParams"`
		ContinuationCommand struct {
			Token string `json:"token"`
		} `json:"continuationCommand"`
	} `json:"continuationEndpoint"`
}

type ContinuationPostData struct {
	Context      ContinuationContext `json:"context"`
	Continuation string              `json:"continuation"`
}

type ContinuationContext struct {
	Client        ContinuationClient        `json:"client"`
	ClickTracking ContinuationClickTracking `json:"clickTracking"`
}

type ContinuationClient struct {
	VisitorData   string `json:"visitorData"`
	ClientName    string `json:"clientName"`
	ClientVersion string `json:"clientVersion"`
}

type ContinuationClickTracking struct {
	ClickTrackingParams string `json:"clickTrackingParams"`
}

type GridVideo struct {
	VideoID   string `json:"videoId"`
	Thumbnail struct {
		Thumbnails []struct {
			URL    string `json:"url"`
			Width  int    `json:"width"`
			Height int    `json:"height"`
		} `json:"thumbnails"`
		WebThumbnailDetailsExtensionData struct {
			IsPreloaded bool `json:"isPreloaded"`
		} `json:"webThumbnailDetailsExtensionData"`
	} `json:"thumbnail"`
	Title struct {
		Runs []struct {
			Text string `json:"text"`
		} `json:"runs"`
	} `json:"title"`
	LongBylineText struct {
		Runs []struct {
			Text               string `json:"text"`
			NavigationEndpoint struct {
				ClickTrackingParams string `json:"clickTrackingParams"`
				CommandMetadata     struct {
					WebCommandMetadata struct {
						URL string `json:"url"`
					} `json:"webCommandMetadata"`
				} `json:"commandMetadata"`
				BrowseEndpoint struct {
					BrowseID string `json:"browseId"`
				} `json:"browseEndpoint"`
			} `json:"navigationEndpoint"`
		} `json:"runs"`
	} `json:"longBylineText"`
	ViewCountText struct {
		Runs []struct {
			Text string `json:"text"`
		} `json:"runs"`
	} `json:"viewCountText"`
	NavigationEndpoint struct {
		ClickTrackingParams string `json:"clickTrackingParams"`
		CommandMetadata     struct {
			WebCommandMetadata struct {
				URL string `json:"url"`
			} `json:"webCommandMetadata"`
		} `json:"commandMetadata"`
		WatchEndpoint struct {
			VideoID string `json:"videoId"`
		} `json:"watchEndpoint"`
	} `json:"navigationEndpoint"`
	ShortBylineText struct {
		Runs []struct {
			Text               string `json:"text"`
			NavigationEndpoint struct {
				ClickTrackingParams string `json:"clickTrackingParams"`
				CommandMetadata     struct {
					WebCommandMetadata struct {
						URL string `json:"url"`
					} `json:"webCommandMetadata"`
				} `json:"commandMetadata"`
				BrowseEndpoint struct {
					BrowseID string `json:"browseId"`
				} `json:"browseEndpoint"`
			} `json:"navigationEndpoint"`
		} `json:"runs"`
	} `json:"shortBylineText"`
	ChannelThumbnail struct {
		Thumbnails []struct {
			URL    string `json:"url"`
			Width  int    `json:"width"`
			Height int    `json:"height"`
		} `json:"thumbnails"`
	} `json:"channelThumbnail"`
	TrackingParams     string `json:"trackingParams"`
	ShortViewCountText struct {
		Runs []struct {
			Text string `json:"text"`
		} `json:"runs"`
	} `json:"shortViewCountText"`
	Menu struct {
		MenuRenderer struct {
			Items []struct {
				MenuNavigationItemRenderer struct {
					Text struct {
						Runs []struct {
							Text string `json:"text"`
						} `json:"runs"`
					} `json:"text"`
					NavigationEndpoint struct {
						ClickTrackingParams string `json:"clickTrackingParams"`
						CommandMetadata     struct {
							WebCommandMetadata struct {
								URL string `json:"url"`
							} `json:"webCommandMetadata"`
						} `json:"commandMetadata"`
						SignInEndpoint struct {
							Hack bool `json:"hack"`
						} `json:"signInEndpoint"`
					} `json:"navigationEndpoint"`
					TrackingParams string `json:"trackingParams"`
				} `json:"menuNavigationItemRenderer"`
			} `json:"items"`
			TrackingParams string `json:"trackingParams"`
			Accessibility  struct {
				AccessibilityData struct {
					Label string `json:"label"`
				} `json:"accessibilityData"`
			} `json:"accessibility"`
		} `json:"menuRenderer"`
	} `json:"menu"`
	ThumbnailOverlays []struct {
		ThumbnailOverlayTimeStatusRenderer struct {
			Text struct {
				Runs []struct {
					Text string `json:"text"`
				} `json:"runs"`
				Accessibility struct {
					AccessibilityData struct {
						Label string `json:"label"`
					} `json:"accessibilityData"`
				} `json:"accessibility"`
			} `json:"text"`
			Style string `json:"style"`
		} `json:"thumbnailOverlayTimeStatusRenderer"`
	} `json:"thumbnailOverlays"`
}

type Continuation struct {
	NextContinuationData *struct {
		Continuation        string `json:"continuation"`
		ClickTrackingParams string `json:"clickTrackingParams"`
	} `json:"nextContinuationData"`
	ReloadContinuationData struct {
		Continuation        string `json:"continuation"`
		ClickTrackingParams string `json:"clickTrackingParams"`
	} `json:"reloadContinuationData"`
}

type GridRenderer struct {
	Items            []GameAndContinuationItem `json:"items"`
	TrackingParams   string                    `json:"trackingParams"`
	VisibleItemCount int                       `json:"visibleItemCount"`
	Continuations    []Continuation            `json:"continuations"`
}

type CompactVideoRenderer struct {
	VideoID   string `json:"videoId"`
	Thumbnail struct {
		Thumbnails []struct {
			URL    string `json:"url"`
			Width  int    `json:"width"`
			Height int    `json:"height"`
		} `json:"thumbnails"`
	} `json:"thumbnail"`
	Title struct {
		Runs []struct {
			Text string `json:"text"`
		} `json:"runs"`
		Accessibility struct {
			AccessibilityData struct {
				Label string `json:"label"`
			} `json:"accessibilityData"`
		} `json:"accessibility"`
	} `json:"title"`
	LongBylineText struct {
		Runs []struct {
			Text               string `json:"text"`
			NavigationEndpoint struct {
				ClickTrackingParams string `json:"clickTrackingParams"`
				CommandMetadata     struct {
					WebCommandMetadata struct {
						URL         string `json:"url"`
						WebPageType string `json:"webPageType"`
						RootVe      int    `json:"rootVe"`
					} `json:"webCommandMetadata"`
				} `json:"commandMetadata"`
				BrowseEndpoint struct {
					BrowseID         string `json:"browseId"`
					CanonicalBaseURL string `json:"canonicalBaseUrl"`
				} `json:"browseEndpoint"`
			} `json:"navigationEndpoint"`
		} `json:"runs"`
	} `json:"longBylineText"`
	ViewCountText struct {
		Runs []struct {
			Text string `json:"text"`
		} `json:"runs"`
	} `json:"viewCountText"`
	NavigationEndpoint struct {
		ClickTrackingParams string `json:"clickTrackingParams"`
		CommandMetadata     struct {
			WebCommandMetadata struct {
				URL         string `json:"url"`
				WebPageType string `json:"webPageType"`
				RootVe      int    `json:"rootVe"`
			} `json:"webCommandMetadata"`
		} `json:"commandMetadata"`
		WatchEndpoint struct {
			VideoID string `json:"videoId"`
		} `json:"watchEndpoint"`
	} `json:"navigationEndpoint"`
	ShortBylineText struct {
		Runs []struct {
			Text               string `json:"text"`
			NavigationEndpoint struct {
				ClickTrackingParams string `json:"clickTrackingParams"`
				CommandMetadata     struct {
					WebCommandMetadata struct {
						URL         string `json:"url"`
						WebPageType string `json:"webPageType"`
						RootVe      int    `json:"rootVe"`
					} `json:"webCommandMetadata"`
				} `json:"commandMetadata"`
				BrowseEndpoint struct {
					BrowseID         string `json:"browseId"`
					CanonicalBaseURL string `json:"canonicalBaseUrl"`
				} `json:"browseEndpoint"`
			} `json:"navigationEndpoint"`
		} `json:"runs"`
	} `json:"shortBylineText"`
	ChannelThumbnail struct {
		Thumbnails []struct {
			URL    string `json:"url"`
			Width  int    `json:"width"`
			Height int    `json:"height"`
		} `json:"thumbnails"`
	} `json:"channelThumbnail"`
	OwnerBadges []struct {
		MetadataBadgeRenderer struct {
			Icon struct {
				IconType string `json:"iconType"`
			} `json:"icon"`
			Style          string `json:"style"`
			Tooltip        string `json:"tooltip"`
			TrackingParams string `json:"trackingParams"`
		} `json:"metadataBadgeRenderer"`
	} `json:"ownerBadges"`
	TrackingParams     string `json:"trackingParams"`
	ShortViewCountText struct {
		Runs []struct {
			Text string `json:"text"`
		} `json:"runs"`
	} `json:"shortViewCountText"`
	Menu struct {
		MenuRenderer struct {
			Items []struct {
				MenuNavigationItemRenderer struct {
					Text struct {
						Runs []struct {
							Text string `json:"text"`
						} `json:"runs"`
					} `json:"text"`
					NavigationEndpoint struct {
						ClickTrackingParams string `json:"clickTrackingParams"`
						CommandMetadata     struct {
							WebCommandMetadata struct {
								URL         string `json:"url"`
								WebPageType string `json:"webPageType"`
								RootVe      int    `json:"rootVe"`
							} `json:"webCommandMetadata"`
						} `json:"commandMetadata"`
						SignInEndpoint struct {
							Hack bool `json:"hack"`
						} `json:"signInEndpoint"`
					} `json:"navigationEndpoint"`
					TrackingParams string `json:"trackingParams"`
				} `json:"menuNavigationItemRenderer"`
			} `json:"items"`
			TrackingParams string `json:"trackingParams"`
			Accessibility  struct {
				AccessibilityData struct {
					Label string `json:"label"`
				} `json:"accessibilityData"`
			} `json:"accessibility"`
		} `json:"menuRenderer"`
	} `json:"menu"`
	ThumbnailOverlays []struct {
		ThumbnailOverlayTimeStatusRenderer struct {
			Text struct {
				Runs []struct {
					Text string `json:"text"`
				} `json:"runs"`
				Accessibility struct {
					AccessibilityData struct {
						Label string `json:"label"`
					} `json:"accessibilityData"`
				} `json:"accessibility"`
			} `json:"text"`
			Style string `json:"style"`
		} `json:"thumbnailOverlayTimeStatusRenderer"`
	} `json:"thumbnailOverlays"`
	Accessibility struct {
		AccessibilityData struct {
			Label string `json:"label"`
		} `json:"accessibilityData"`
	} `json:"accessibility"`
}

type ItemSectionRenderer struct {
	TrackingParams string `json:"trackingParams"`
	Header         struct {
		SortFilterHeaderRenderer struct {
			FilterMenu struct {
				SortFilterSubMenuRenderer struct {
					SubMenuItems []struct {
						Title        string       `json:"title"`
						Selected     bool         `json:"selected"`
						Continuation Continuation `json:"continuation"`
					} `json:"subMenuItems"`
					TrackingParams string `json:"trackingParams"`
				} `json:"sortFilterSubMenuRenderer"`
			} `json:"filterMenu"`
			TrackingParams string `json:"trackingParams"`
		} `json:"sortFilterHeaderRenderer"`
	} `json:"header"`
	Contents []struct {
		ShelfRenderer struct {
			Content struct {
				GridRenderer           GridRenderer `json:"gridRenderer"`
				HorizontalListRenderer struct {
					Items []struct {
						GridVideoRenderer GridVideo `json:"gridVideoRenderer"`
					} `json:"items"`
					TrackingParams   string `json:"trackingParams"`
					VisibleItemCount int    `json:"visibleItemCount"`
				} `json:"horizontalListRenderer"`
			} `json:"content"`
			SortFilter struct {
				SortFilterSubMenuRenderer struct {
					SubMenuItems []struct {
						Title        string       `json:"title"`
						Selected     bool         `json:"selected"`
						Continuation Continuation `json:"continuation"`
					} `json:"subMenuItems"`
					TrackingParams string `json:"trackingParams"`
				} `json:"sortFilterSubMenuRenderer"`
			} `json:"sortFilter"`
			TrackingParams string `json:"trackingParams"`
			HideHeader     bool   `json:"hideHeader"`
		} `json:"shelfRenderer"`
		CompactVideoRenderer CompactVideoRenderer `json:"compactVideoRenderer,omitempty"`
	} `json:"contents"`
	Continuations []Continuation `json:"continuations"`
}

type SectionList struct {
	Contents []struct {
		ShelfRenderer struct {
			Content struct {
				GridRenderer           GridRenderer `json:"gridRenderer"`
				HorizontalListRenderer struct {
					Items []struct {
						GridVideoRenderer GridVideo `json:"gridVideoRenderer"`
					} `json:"items"`
					TrackingParams   string `json:"trackingParams"`
					VisibleItemCount int    `json:"visibleItemCount"`
				} `json:"horizontalListRenderer"`
			} `json:"content"`
			TrackingParams string `json:"trackingParams"`
			HideHeader     bool   `json:"hideHeader"`
		} `json:"shelfRenderer"`
		ItemSectionRenderer ItemSectionRenderer `json:"itemSectionRenderer"`
	} `json:"contents"`
	Continuations  []Continuation `json:"continuations"`
	TrackingParams string         `json:"trackingParams"`
}

type ApiResponse struct {
	Csn       string `json:"csn"`
	XSRFToken string `json:"xsrf_token"`
	Response  struct {
		ResponseContext struct {
			WebResponseContextExtensionData struct {
				YtConfigData struct {
					VisitorData string `json:"visitorData"`
				} `json:"ytConfigData"`
			} `json:"webResponseContextExtensionData"`
		} `json:"responseContext"`
		Contents struct {
			SingleColumnBrowseResultsRenderer struct {
				Tabs []struct {
					TabRenderer struct {
						Title    string `json:"title"`
						Selected bool   `json:"selected"`
						Content  struct {
							SectionListRenderer SectionList `json:"sectionListRenderer"`
						} `json:"content"`
						TrackingParams string `json:"trackingParams"`
					} `json:"tabRenderer"`
				} `json:"tabs"`
				HideTabBar bool `json:"hideTabBar"`
			} `json:"singleColumnBrowseResultsRenderer"`
			TwoColumnBrowseResultsRenderer struct {
				Tabs []struct {
					TabRenderer struct {
						Title   string `json:"title"`
						Content struct {
							SectionListRenderer SectionList `json:"sectionListRenderer"`
						} `json:"content"`
						TabIdentifier  string `json:"tabIdentifier"`
						TrackingParams string `json:"trackingParams"`
					} `json:"tabRenderer,omitempty"`
				} `json:"tabs"`
			} `json:"twoColumnBrowseResultsRenderer"`
		} `json:"contents"`
		ContinuationContents struct {
			SectionListContinuation SectionList         `json:"sectionListContinuation"`
			GridContinuation        GridRenderer        `json:"gridContinuation"`
			ItemSectionContinuation ItemSectionRenderer `json:"itemSectionContinuation"`
		} `json:"continuationContents"`
		TrackingParams string `json:"trackingParams"`
	} `json:"response"`
	URL  string `json:"url"`
	Page string `json:"page"`
}

type ContinuationResponse struct {
	OnResponseReceivedActions []struct {
		AppendContinuationItemsAction struct {
			ContinuationItems []GameAndContinuationItem `json:"continuationItems"`
		} `json:"appendContinuationItemsAction"`
	} `json:"onResponseReceivedActions"`
}

func (c *YoutubeCrawler) apiCall(API string, query *url.Values) (*http.Response, error) {
	req, err := http.NewRequest("GET", API, nil)
	if query == nil {
		query = &url.Values{}
	}
	query.Add("pbj", "1")
	req.URL.RawQuery = query.Encode()
	if err != nil {
		return nil, err
	}
	req.Header.Set("User-Agent", util.GetMobileUserAgent())
	req.Header.Set("X-YouTube-Client-Name", "2")
	req.Header.Set("X-YouTube-Client-Version", "2.20200925.01.00")
	req.Header.Set("Cookie", c.cookies)

	resp, err := c.HTTPClient.Do(req)

	if err != nil {
		return nil, fmt.Errorf("apiCall failed to fetch: %s", err.Error())
	}

	if resp.StatusCode != 200 {
		return nil, fmt.Errorf("Request failed: %d %s", resp.StatusCode, resp.Status)
	}

	return resp, nil
}

func (c *YoutubeCrawler) apiCallWeb(API string, query *url.Values) (*http.Response, error) {
	req, err := http.NewRequest("GET", API, nil)
	if query == nil {
		query = &url.Values{}
	}
	query.Add("pbj", "1")
	req.URL.RawQuery = query.Encode()
	if err != nil {
		return nil, err
	}
	req.Header.Set("User-Agent", util.GetRandomUserAgent())
	req.Header.Set("X-YouTube-Client-Name", "1")
	req.Header.Set("X-YouTube-Client-Version", "2.20200925.01.00")
	req.Header.Set("Cookie", c.cookies)

	resp, err := c.HTTPClient.Do(req)

	if err != nil {
		return nil, fmt.Errorf("apiCall failed to fetch: %s", err.Error())
	}

	if resp.StatusCode != 200 {
		return nil, fmt.Errorf("Request failed: %d %s", resp.StatusCode, resp.Status)
	}

	return resp, nil
}

func (c *YoutubeCrawler) gameContinuationCall(API string, postData io.Reader) (*http.Response, error) {
	req, err := http.NewRequest("POST", API, postData)

	if err != nil {
		return nil, err
	}
	req.Header.Set("User-Agent", util.GetRandomUserAgent())
	req.Header.Set("X-YouTube-Client-Name", "1")
	req.Header.Set("X-YouTube-Client-Version", "2.20200925.01.00")
	req.Header.Set("Cookie", c.cookies)

	resp, err := c.HTTPClient.Do(req)

	if err != nil {
		return nil, fmt.Errorf("apiCall failed to fetch: %s", err.Error())
	}

	if resp.StatusCode != 200 {
		return nil, fmt.Errorf("Request failed: %d %s", resp.StatusCode, resp.Status)
	}

	return resp, nil
}

func (c *YoutubeCrawler) getFirstGamesWeb() (*GridRenderer, error) {
	resp, err := c.apiCallWeb(gameAPIWeb, nil)
	if err != nil {
		return nil, fmt.Errorf("getFirstGame failed to load: %s", err.Error())
	}
	defer resp.Body.Close()
	dataJSON := []ApiResponse{}
	err = json.NewDecoder(resp.Body).Decode(&dataJSON)
	if err != nil {
		return nil, fmt.Errorf("getFirstGame failed to decode: %s", err.Error())
	}

	// yt now uses multiple cookies
	cookies := resp.Header.Values("Set-Cookie")
	cks := []string{}
	for _, ck := range cookies {
		splits := strings.Split(ck, ";")
		cks = append(cks, splits[0])
	}
	c.cookies = strings.Join(cks, ";")

	if len(dataJSON) < 2 {
		return nil, fmt.Errorf("not enough resp")
	}
	if len(dataJSON[1].Response.Contents.TwoColumnBrowseResultsRenderer.Tabs) == 0 {
		return nil, fmt.Errorf("getFirstGames not enough tabs")
	}
	if len(dataJSON[1].Response.Contents.TwoColumnBrowseResultsRenderer.Tabs[0].TabRenderer.Content.SectionListRenderer.Contents) == 0 {
		return nil, fmt.Errorf("not enough contents")
	}
	if len(dataJSON[1].Response.Contents.TwoColumnBrowseResultsRenderer.Tabs[0].TabRenderer.Content.SectionListRenderer.Contents[0].ItemSectionRenderer.Contents) == 0 {
		return nil, fmt.Errorf("not enough contents")
	}
	visitorData = dataJSON[1].Response.ResponseContext.WebResponseContextExtensionData.YtConfigData.VisitorData
	return &dataJSON[1].Response.Contents.TwoColumnBrowseResultsRenderer.Tabs[0].TabRenderer.Content.SectionListRenderer.Contents[0].ItemSectionRenderer.Contents[0].ShelfRenderer.Content.GridRenderer, nil
}

func (c *YoutubeCrawler) getNextGamesWeb(ctoken string, itct string) (*GridRenderer, error) {
	q := url.Values{}
	q.Add("ctoken", ctoken)
	q.Add("continuation", ctoken)
	q.Add("itct", itct)
	resp, err := c.apiCallWeb(ajaxAPI, &q)
	if err != nil {
		return nil, fmt.Errorf("getNextGame failed to load: %s", err.Error())
	}

	dataJSON := []ApiResponse{}
	err = json.NewDecoder(resp.Body).Decode(&dataJSON)
	if err != nil {
		return nil, fmt.Errorf("getNextGame failed to decode: %s", err.Error())
	}

	if len(dataJSON) < 2 {
		return nil, fmt.Errorf("not enough resp")
	}
	return &dataJSON[1].Response.ContinuationContents.GridContinuation, nil
}

func (c *YoutubeCrawler) getNextGames(ctoken string, itct string) ([]GameAndContinuationItem, error) {
	if visitorData == "" {
		return nil, errors.New("VisitorData is empty, please call getFirstGameWeb first or check getFirstGameWeb http response")
	}
	postData := ContinuationPostData{
		Context: ContinuationContext{
			Client: ContinuationClient{
				VisitorData:   visitorData,
				ClientName:    "WEB",
				ClientVersion: "2.20200925.01.0",
			},
			ClickTracking: ContinuationClickTracking{
				ClickTrackingParams: itct,
			},
		},
		Continuation: ctoken,
	}
	postDataBuffer := new(bytes.Buffer)
	json.NewEncoder(postDataBuffer).Encode(postData)
	resp, err := c.gameContinuationCall(youtubeAPI, postDataBuffer)
	if err != nil {
		return nil, fmt.Errorf("getNextGames failed to load: %s", err.Error())
	}

	dataJSON := ContinuationResponse{}
	err = json.NewDecoder(resp.Body).Decode(&dataJSON)
	if err != nil {
		return nil, fmt.Errorf("getNextGame failed to decode: %s", err.Error())
	}

	if len(dataJSON.OnResponseReceivedActions) <= 0 {
		return nil, errors.New("getNextGames failed to load, no OnResponseReceivedActions in response")
	}

	return dataJSON.OnResponseReceivedActions[0].AppendContinuationItemsAction.ContinuationItems, nil
}

func (c *YoutubeCrawler) getWorldWidePage(gameID string) (*Continuation, error) {
	if gameID == "" {
		return nil, fmt.Errorf("empty game id")
	}
	resp, err := c.apiCallWeb(fmt.Sprintf(liveWebAPI, gameID), nil)
	if err != nil {
		return nil, fmt.Errorf("getWorldWidePage failed to load: %s", err.Error())
	}
	defer resp.Body.Close()
	dataJSON := []ApiResponse{}
	err = json.NewDecoder(resp.Body).Decode(&dataJSON)
	if err != nil {
		return nil, fmt.Errorf("getWorldWidePage failed to decode: %s", err.Error())
	}
	if len(dataJSON) < 2 {
		return nil, fmt.Errorf("getWorldWidePage not enough response")
	}
	if len(dataJSON[1].Response.Contents.TwoColumnBrowseResultsRenderer.Tabs) < 2 {
		return nil, fmt.Errorf("getWorldWidePage not enough tabs")
	}
	if len(dataJSON[1].Response.Contents.TwoColumnBrowseResultsRenderer.Tabs[1].TabRenderer.Content.SectionListRenderer.Contents) < 1 {
		return nil, fmt.Errorf("getWorldWidePage not enough contents")
	}

	items := dataJSON[1].Response.Contents.TwoColumnBrowseResultsRenderer.Tabs[1].TabRenderer.Content.SectionListRenderer.Contents[0].ItemSectionRenderer.Contents[0].ShelfRenderer.SortFilter.SortFilterSubMenuRenderer.SubMenuItems
	for _, item := range items {
		if item.Title == "Worldwide" {
			return &item.Continuation, nil
		}
	}
	return nil, fmt.Errorf("getWorldWidePage did not find worldwide page")
}

func (c *YoutubeCrawler) getFirstPage(gameID string) (*ItemSectionRenderer, error) {
	if gameID == "" {
		return nil, fmt.Errorf("getFirstPage empty game id")
	}
	resp, err := c.apiCall(fmt.Sprintf(liveAPI, gameID), nil)
	if err != nil {
		return nil, fmt.Errorf("getFirstPage failed to load: %s", err.Error())
	}
	defer resp.Body.Close()
	dataJSON := ApiResponse{}
	err = json.NewDecoder(resp.Body).Decode(&dataJSON)
	if err != nil {
		return nil, fmt.Errorf("getFirstPage failed to decode: %s", err.Error())
	}
	if len(dataJSON.Response.Contents.SingleColumnBrowseResultsRenderer.Tabs) < 2 {
		return nil, fmt.Errorf("getFirstPage not enough tabs")
	}
	if len(dataJSON.Response.Contents.SingleColumnBrowseResultsRenderer.Tabs[1].TabRenderer.Content.SectionListRenderer.Contents) == 0 {
		return nil, fmt.Errorf("getFirstPage not enough content")
	}
	return &dataJSON.Response.Contents.SingleColumnBrowseResultsRenderer.Tabs[1].TabRenderer.Content.SectionListRenderer.Contents[0].ItemSectionRenderer, nil
}

func (c *YoutubeCrawler) getWorldWideGame(gameID string, ctoken string, itct string) (*ItemSectionRenderer, error) {
	q := url.Values{}
	q.Add("ctoken", ctoken)
	q.Add("itct", itct)
	resp, err := c.apiCall(fmt.Sprintf(liveAPI, gameID), &q)
	if err != nil {
		return nil, fmt.Errorf("getNextPage failed to load: %s", err.Error())
	}
	defer resp.Body.Close()
	dataJSON := ApiResponse{}
	err = json.NewDecoder(resp.Body).Decode(&dataJSON)
	if err != nil {
		return nil, fmt.Errorf("getNextPage failed to decode: %s", err.Error())
	}
	if len(dataJSON.Response.ContinuationContents.SectionListContinuation.Contents) == 0 {
		return nil, fmt.Errorf("not enough contents")
	}
	return &dataJSON.Response.ContinuationContents.SectionListContinuation.Contents[0].ItemSectionRenderer, nil
}

func (c *YoutubeCrawler) getNextPage(gameID string, ctoken string, itct string) (*ItemSectionRenderer, error) {
	q := url.Values{}
	q.Add("ctoken", ctoken)
	q.Add("itct", itct)
	resp, err := c.apiCall(fmt.Sprintf(liveAPI, gameID), &q)
	if err != nil {
		return nil, fmt.Errorf("getNextPage failed to load: %s", err.Error())
	}
	defer resp.Body.Close()
	dataJSON := ApiResponse{}
	err = json.NewDecoder(resp.Body).Decode(&dataJSON)
	if err != nil {
		return nil, fmt.Errorf("getNextPage failed to decode: %s", err.Error())
	}
	return &dataJSON.Response.ContinuationContents.ItemSectionContinuation, nil
}

func (c *YoutubeCrawler) processLiveRoom(liveSection *ItemSectionRenderer, gameTitle string, timestamp int64, results chan<- model.ChannelInfo) {
	for _, stream := range liveSection.Contents {
		if stream.CompactVideoRenderer.VideoID == "" {
			continue
		}
		if len(stream.CompactVideoRenderer.LongBylineText.Runs) > 0 {
			var channelTitle = ""
			if len(stream.CompactVideoRenderer.Title.Runs) != 0 {
				channelTitle = stream.CompactVideoRenderer.Title.Runs[0].Text
			}
			var lang string
			if langInfo := whatlanggo.Detect(channelTitle); langInfo.Confidence > 0.5 {
				lang = langInfo.Lang.String()
			}
			channelName := stream.CompactVideoRenderer.LongBylineText.Runs[0].Text

			var ccv uint64
			viewCountTextRuns := stream.CompactVideoRenderer.ViewCountText.Runs
			if len(viewCountTextRuns) > 0 {
				viewerText := viewCountTextRuns[0].Text
				var err error
				countContents := strings.Split(viewerText, " ")
				if (len(countContents) > 1 && countContents[1] != Watching) ||
					(len(viewCountTextRuns) > 1 && strings.TrimSpace(viewCountTextRuns[1].Text) != Watching) {
					c.Log.Debugf("Skip channel [%s], because it's viewer state is not [watching]", channelName)
					continue
				}

				ccv, err = humanize.ParseBytes(countContents[0])
				if err != nil {
					c.Log.Warningf("Cannot parse ccv for %s: %s", channelName, err.Error())
				}
			}

			results <- model.ChannelInfo{
				ChannelID:    stream.CompactVideoRenderer.LongBylineText.Runs[0].NavigationEndpoint.CommandMetadata.WebCommandMetadata.URL,
				ChannelName:  channelName,
				ChannelTitle: channelTitle,
				CountryCode:  "", // TODO
				Game:         gameTitle,
				Language:     lang,
				Platform:     c.Platform(),
				Ccv:          int64(ccv),
				TimeCrawled:  timestamp,
			}
		} else {
			c.Log.Debugf("no channel name %v", stream.CompactVideoRenderer.LongBylineText.Runs)
		}
	}
}

func getContinuation(conts []Continuation) *Continuation {
	for _, cont := range conts {
		if cont.NextContinuationData != nil {
			return &cont
		}
	}
	return nil
}

func (c *YoutubeCrawler) Crawl() error {
	timestamp := time.Now().Unix()

	var games []GameCard
	totalViewers := uint64(0)
	gameSection, err := c.getFirstGamesWeb()
	gameAndContItems := gameSection.Items
	for {
		if err != nil {
			c.Log.Errorf("rip query: %s", err.Error())
			break
		}

		if len(gameAndContItems) == 0 {
			c.Log.Error("didn't load any games!")
			break
		}

		c.Log.Debugf("loaded %d games", len(gameAndContItems))

		for _, game := range gameAndContItems {
			// if len(game.GameCardRenderer.Game.GameDetailsRenderer.Title.Runs) == 0 {
			// 	c.Log.Warningf("Cannot find title for %s", game.GameCardRenderer.Game.GameDetailsRenderer.Title.Runs)
			// 	continue
			// }

			gameTitle := game.GameCardRenderer.Game.GameDetailsRenderer.Title.SimpleText
			if gameTitle == "" {
				continue
			}
			ccv := uint64(0)

			if len(game.GameCardRenderer.Game.GameDetailsRenderer.LiveViewersText.Runs) == 0 {
				c.Log.Warningf("Cannot find ccv for %s %s", gameTitle, game.GameCardRenderer.Game.GameDetailsRenderer.LiveViewersText.Runs)
			} else {
				viewerText := game.GameCardRenderer.Game.GameDetailsRenderer.LiveViewersText.Runs[0].Text
				ccv, err = humanize.ParseBytes(strings.Split(viewerText, " ")[0])
				if err != nil {
					c.Log.Warningf("Cannot parse ccv for %s %v", gameTitle, err)
				}
			}

			c.Log.Debugf("%s: %d", gameTitle, ccv)
			games = append(games, game.GameCardRenderer)
			totalViewers += ccv
		}

		// fetch next page
		cont := getContinuation(gameSection.Continuations)
		if cont == nil {
			continuationItem := gameAndContItems[len(gameAndContItems)-1].ContinuationItemRenderer
			if continuationItem == nil {
				break
			}
			gameAndContItems, err = c.getNextGames(continuationItem.ContinuationEndpoint.ContinuationCommand.Token,
				continuationItem.ContinuationEndpoint.ClickTrackingParams)
			continue
		}
		ctoken := cont.NextContinuationData.Continuation
		itct := cont.NextContinuationData.ClickTrackingParams
		time.Sleep(200 * time.Millisecond)
		gameSection, err = c.getNextGamesWeb(ctoken, itct)
		gameAndContItems = gameSection.Items
	}

	c.Log.Infof("loaded %d game categories with %d viewers", len(games), totalViewers)

	results := make(chan model.ChannelInfo, len(games)*100)
	done := make(chan bool)
	go func() {
		c.TrackChannelInfo(results, totalViewers)
		done <- true
	}()

	wg := &sync.WaitGroup{}
	for _, game := range games {
		wg.Add(1)
		go func(game GameCard) {
			defer wg.Done()
			gameTitle := game.Game.GameDetailsRenderer.Title.SimpleText
			gameID := game.Game.GameDetailsRenderer.Endpoint.BrowseEndpoint.BrowseID

			cont, err := c.getWorldWidePage(gameID)

			var liveSection *ItemSectionRenderer
			if err != nil {
				c.Log.Warningf("Failed to get worldwide page %s", err.Error())
				liveSection, err = c.getFirstPage(gameID)
			} else {
				liveSection, err = c.getWorldWideGame(gameID, cont.ReloadContinuationData.Continuation, cont.ReloadContinuationData.ClickTrackingParams)
			}

			pages := 0
			for {
				if err != nil {
					c.Log.Errorf("rip query %s %v", gameID, err)
					break
				}

				if len(liveSection.Contents) == 0 {
					c.Log.Errorf("didn't load any streams %s %v, gameId: %s", gameTitle, liveSection, gameID)
					break
				}

				if pages == 0 {
					c.Log.Debugf("loaded %s, %d streams", gameTitle, len(liveSection.Contents))
				}

				c.processLiveRoom(liveSection, gameTitle, timestamp, results)

				// fetch next page
				cont := getContinuation(liveSection.Continuations)
				if cont == nil {
					break
				}
				ctoken := cont.NextContinuationData.Continuation
				itct := cont.NextContinuationData.ClickTrackingParams

				time.Sleep(50 * time.Millisecond)
				liveSection, err = c.getNextPage(gameID, ctoken, itct)
				pages++
			}
		}(game)
	}
	wg.Wait()

	close(results)
	<-done

	return nil
}
