package proxy

import (
	"bufio"
	"bytes"
	"compress/gzip"
	"encoding/base64"
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httputil"
	"os"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/lambda"

	"code.justin.tv/esports-exp/marionette/pkg/proxy/server"
)

type ProxyClient struct {
	client       *lambda.Lambda
	functionName string
}

func New() *ProxyClient {
	ls := &ProxyClient{}

	sess := session.Must(session.NewSessionWithOptions(session.Options{
		Config: aws.Config{Region: aws.String(os.Getenv("PROXYSERVER_REGION"))},
	}))

	ls.client = lambda.New(sess)
	ls.functionName = os.Getenv("PROXYSERVER_ENDPOINT")

	return ls
}

func (pc *ProxyClient) Do(req *http.Request) (*http.Response, error) {
	dump, _ := httputil.DumpRequestOut(req, true)

	request := server.Request{Request: string(dump)}

	payload, err := json.Marshal(request)
	if err != nil {
		return nil, fmt.Errorf("error marshalling request: %s", err.Error())
	}

	result, err := pc.client.Invoke(&lambda.InvokeInput{FunctionName: aws.String(pc.functionName), Payload: payload})
	if err != nil {
		return nil, fmt.Errorf("error calling function %s: %s", pc.functionName, err.Error())
	}

	resp := server.Response{}
	err = json.Unmarshal(result.Payload, &resp)
	if err != nil {
		return nil, fmt.Errorf("failed to unmarshall response: [%s]. LambdaInvokeOutput Payload: [%s]", err.Error(), string(result.Payload))
	}

	respRaw, err := base64.StdEncoding.DecodeString(resp.Response)
	if err != nil {
		return nil, fmt.Errorf("failed to decode response: [%s]. LambdaInvokeOutput Payload: [%s]", err.Error(), string(result.Payload))
	}

	proxyResp, err := http.ReadResponse(bufio.NewReader(bytes.NewReader(respRaw)), req)
	if err != nil {
		return nil, fmt.Errorf("failed to parse response: [%s]. LambdaInvokeOutput Payload: [%s]", err.Error(), string(result.Payload))
	}

	if proxyResp.Header.Get("Content-Encoding") == "gzip" {
		reader, err := gzip.NewReader(proxyResp.Body)
		if err != nil {
			return nil, fmt.Errorf("failed to uncompress gzip:[%s]. LambdaInvokeOutput Payload: [%s]", err.Error(), string(result.Payload))
		}
		proxyResp.Body = reader
		proxyResp.Header.Del("Content-Encoding")
	}
	return proxyResp, nil
}
