import cdk = require("@aws-cdk/core");
import ec2 = require("@aws-cdk/aws-ec2");
import ecs = require("@aws-cdk/aws-ecs");
import ecs_patterns = require("@aws-cdk/aws-ecs-patterns");
import events = require("@aws-cdk/aws-events");
import ecr = require("@aws-cdk/aws-ecr");
import events_targets = require("@aws-cdk/aws-events-targets");

class PublicScheduledFargateTask extends ecs_patterns.ScheduledFargateTask {
  addTaskDefinitionToEventTarget(taskDefinition: ecs.FargateTaskDefinition) {
    const eventRuleTarget = new events_targets.EcsTask({
      cluster: this.cluster,
      taskDefinition,
      taskCount: this.desiredTaskCount,
      subnetSelection: { subnetType: ec2.SubnetType.PUBLIC } // Make it public
    });
    this.eventRule.addTarget(eventRuleTarget);
    return eventRuleTarget;
  }
}

let config = {
  interval: 5,
  spadeBlurprint: "marionette_category_test"
};

if (process.env.AWS_PROFILE === "twitch-marionette-prod") {
  config = {
    interval: 1,
    spadeBlurprint: "marionette_category_prod"
  };
}

export class CdkStack extends cdk.Stack {
  constructor(scope: cdk.App, id: string, props?: cdk.StackProps) {
    super(scope, id, props);

    const vpc = new ec2.Vpc(this, "MarionetteChromeVpc", {
      subnetConfiguration: [
        {
          cidrMask: 24,
          name: "public",
          subnetType: ec2.SubnetType.PUBLIC
        }
      ]
    });

    const cluster = new ecs.Cluster(this, "MarionetteChromeCluster", {
      vpc: vpc
    });

    const registry = ecr.Repository.fromRepositoryName(
      this,
      "MarionetteChromeEcr",
      "marionette-chrome"
    );

    // Create a load-balanced Fargate service and make it public
    const fargetTask = new PublicScheduledFargateTask(
      this,
      "MarionetteChromeService",
      {
        cluster: cluster, // Required
        image: ecs.ContainerImage.fromEcrRepository(
          registry,
          process.env.DOCKER_TAG || "latest"
        ), // Required
        cpu: 512, // Default is 256
        desiredTaskCount: 1, // Default is 1
        memoryLimitMiB: 1024, // Default is 512
        schedule: events.Schedule.rate(
          cdk.Duration.minutes(config["interval"])
        ),
        environment: {
          SPADE_BLUEPRINT: config["spadeBlurprint"]
        },
        logDriver: new ecs.AwsLogDriver({
          streamPrefix: "MarionetteChrome"
        })
      }
    );
  }
}
