import os
import sys
import shutil
import zipfile
import urllib.request as req
import subprocess

DIR_PATH = os.path.dirname(os.path.realpath(__file__))
BUILD_DIR = os.path.join(DIR_PATH, 'build')
DIST_DIR = os.path.join(DIR_PATH, 'dist')
DOWNLOAD_DIR = os.path.join(DIR_PATH, 'download')
BUILD_IGNORE = ['build', 'dist', '.git', '.idea', 'download', 'build.py', 'tools']

PYTHON_DIR = 'python'
GST_PLUGIN_DIR = "%s/lib/gstreamer-1.0" % PYTHON_DIR

# If you want to use gst-gl plugins from the compiled gst-launch-1.0
# Swap `gst-bebo` with with this!
# ('gst-bebo/gst-bebo_jake-20181215144056.zip', PYTHON_DIR),

DEPENDENCIES = [
    ('bebo-capture/bebo-capture_master-20180424172439.zip', 'bebodlls'),
    ('gst-bebo/gst-bebo_gst_1_15-20180820184054.zip', PYTHON_DIR),
    ('gst-dshowsrc/gst-dshowsrc_gst_1_15-20180809211509.zip', GST_PLUGIN_DIR),
    ('gst-plugins-bebo/gst-plugins-bebo_master-20190116170547.zip', 'bebodlls'),
    ('gst-gamecapture/gst-gamecapture_master-20190207034306.zip', 'bebodlls'),
    ('gst-rtmp2/gst-rtmp2_gst_1_15-20180809213502.zip', 'bebodlls'),
    ('gst-wasapi/gst-wasapi_gst_1_15-20180827204014.zip',  'bebodlls'),
    ('gst-cef/gst-cef_master-20181012163836.zip', PYTHON_DIR),
    # ('nvdec/libgstnvdec.zip', GST_PLUGIN_DIR)
]


def download(dep):
    download_location = os.path.join(DOWNLOAD_DIR, dep[0].split('/')[-1])
    s3_file_path = 'https://s3-us-west-1.amazonaws.com/bebo-app/repo/' + dep[0]
    print("Downloading", s3_file_path)
    req.urlretrieve(s3_file_path, download_location)
    zip = zipfile.ZipFile(download_location)
    output_location = os.path.join(BUILD_DIR, dep[1])
    print("Extracting {} to {}".format(download_location, output_location))
    zip.extractall(output_location)


def init_build_dir():
    if os.path.exists(BUILD_DIR):
        print("Removing %s ..." % BUILD_DIR)
        shutil.rmtree(BUILD_DIR)
    os.makedirs(BUILD_DIR)
    if os.path.exists(DIST_DIR):
        print("Removing %s ..." % DIST_DIR)
        shutil.rmtree(DIST_DIR)
    os.makedirs(DIST_DIR)

    if os.path.exists(DOWNLOAD_DIR):
        shutil.rmtree(DOWNLOAD_DIR)
    os.makedirs(DOWNLOAD_DIR)


def copytree(src, dst, symlinks=False, ignore=None):
    for item in os.listdir(src):
        if any(dir for dir in BUILD_IGNORE if item in BUILD_IGNORE):
            print(("Ignoring: " + item))
            continue
        s = os.path.join(src, item)
        d = os.path.join(dst, item)
        if os.path.isdir(s):
            shutil.copytree(s, d, symlinks, ignore)
        else:
            shutil.copy2(s, d)


def post_build_cleanup():
    shutil.rmtree(DOWNLOAD_DIR)


def copy_gst_dlls():
    """ only for things which need to go into two places or overwrites an existing plugin, otherwise the
        download takes care of it"""

    # cef files need to be in the same directory that gstlaunch.exe is in or be
    # in the working directory.
    pluginloc = os.path.join(BUILD_DIR, PYTHON_DIR, 'libgstcef.dll')
    newloc = os.path.join(BUILD_DIR, GST_PLUGIN_DIR, 'libgstcef.dll')
    os.rename(pluginloc, newloc)

    # print("Copying custom libgstwasapi...")
    shutil.copy2(os.path.join(BUILD_DIR, 'bebodlls/libgstwasapi.dll'),
                 os.path.join(BUILD_DIR, GST_PLUGIN_DIR))
    shutil.copy2(os.path.join(BUILD_DIR, 'bebodlls/libgstwasapi.pdb'),
                 os.path.join(BUILD_DIR, GST_PLUGIN_DIR))
    os.remove(os.path.join(BUILD_DIR, 'bebodlls/libgstwasapi.dll'))
    os.remove(os.path.join(BUILD_DIR, 'bebodlls/libgstwasapi.pdb'))
    os.remove(os.path.join(BUILD_DIR, GST_PLUGIN_DIR, 'gstwasapi.dll'))
    os.remove(os.path.join(BUILD_DIR, GST_PLUGIN_DIR, 'gstwasapi.pdb'))

    # gst game capture
    print("Copying gst-gamecapture...")
    pluginloc = os.path.join(BUILD_DIR, 'bebodlls/libgstgamecapture.dll')
    newloc = os.path.join(BUILD_DIR, GST_PLUGIN_DIR, 'libgstgamecapture.dll')
    os.rename(pluginloc[:-3] + 'pdb', newloc[:-3] + 'pdb')
    os.rename(pluginloc, newloc)

    print("Copying gst-rtmp2...")
    shutil.copy2(os.path.join(BUILD_DIR, 'bebodlls/libgstrtmp2.dll'),
                 os.path.join(BUILD_DIR, GST_PLUGIN_DIR))
    shutil.copy2(os.path.join(BUILD_DIR, 'bebodlls/libgstrtmp2.pdb'),
                 os.path.join(BUILD_DIR, GST_PLUGIN_DIR))
    os.remove(os.path.join(BUILD_DIR, 'bebodlls/libgstrtmp2.dll'))
    os.remove(os.path.join(BUILD_DIR, 'bebodlls/libgstrtmp2.pdb'))

    print("Copying gst-plugin-bebo")
    shutil.copy2(os.path.join(BUILD_DIR, 'bebodlls/gst-plugin/libgstbebo.dll'),
                 os.path.join(BUILD_DIR, GST_PLUGIN_DIR))
    shutil.copy2(os.path.join(BUILD_DIR, 'bebodlls/gst-plugin/libgstbebo.pdb'),
                 os.path.join(BUILD_DIR, GST_PLUGIN_DIR))
    shutil.rmtree(os.path.join(BUILD_DIR, 'bebodlls/gst-plugin'))

    print("Copying nacl-preview")
    shutil.copy2(os.path.join(BUILD_DIR, 'bebodlls/nacl-plugin/naclpreview.dll'),
                 os.path.join(BUILD_DIR, 'bebodlls'))
    shutil.copy2(os.path.join(BUILD_DIR, 'bebodlls/nacl-plugin/naclpreview.pdb'),
                 os.path.join(BUILD_DIR, 'bebodlls'))
    shutil.rmtree(os.path.join(BUILD_DIR, 'bebodlls/nacl-plugin'))



def main():
    init_build_dir()

    print("Fetching S3 dependencies")
    for dep in DEPENDENCIES:
        download(dep)

    python = os.path.join(BUILD_DIR, PYTHON_DIR, "python.exe")
    bebo_mercy = os.path.join(BUILD_DIR, PYTHON_DIR, "bebo_mercy.exe")
    os.rename(python, bebo_mercy)

    print('Installing requirements.txt')
    result_code = subprocess.call([bebo_mercy, '-m', 'pip', 'install',
                                               '-r', 'requirements.txt'])
    if result_code is not 0:
        print("Failed to install requirements")
        sys.exit(1)

    print('Moving our GStreamer plugins...')
    copy_gst_dlls()

    print('Post build cleanup')
    post_build_cleanup()

    print('Done!')


if __name__ == '__main__':
    main()
