from .branch import Branch, AUDIO_MIXER_LATENCY
from gi.repository import Gst
from classes import logger as log


class AudioMixerInputInfo:
    def __init__(self, ghost_sink_pad, mixer_sink_pad, branch, elements, identifier):
        self.ghost_sink_pad = ghost_sink_pad
        self.mixer_sink_pad = mixer_sink_pad
        self.branch = branch
        self.identifier = identifier
        self.elements = elements


class AudioMixerOutputInfo:
    def __init__(self, ghost_src_pad, tee_src_pad, sink_pad, identifier):
        self.ghost_src_pad = ghost_src_pad
        self.tee_src_pad = tee_src_pad
        self.identifier = identifier
        self.sink_pad = sink_pad


class AudioMixer(Branch):

    def __init__(self, bus, session_id):
        super(AudioMixer, self).__init__(bus, session_id)
        self.mixer_index = 0

        self.identifier_to_input_info = {}
        self.identifier_to_output_info = {}

        audio_mixer = self.make("audiomixer",
                                "audiomixer",
                                latency=AUDIO_MIXER_LATENCY)

        # TODO: we probably don't want to hardcode rate to 48000,
        # we can do it based on the input source
        audio_mixer_caps = self.make("capsfilter",
                                     caps=Gst.Caps.from_string("audio/x-raw,format=F32LE,rate=(int)48000,channels=(int)2"))

        self.tee = self.make("tee")
        self.tee.set_property("allow-not-linked", True)

        self.mixer = audio_mixer

        elements = [audio_mixer, audio_mixer_caps, self.tee]
        self.link_many(*elements)

        before_tee = audio_mixer_caps
        self.block_reconfigure_pad = before_tee.get_static_pad("src")
        self.block_probe_id = self.block_reconfigure_pad.add_probe(
            Gst.PadProbeType.EVENT_UPSTREAM | Gst.PadProbeType.BLOCK,
            self.reconfigure_cb,
            None)

    def reconfigure_cb(self, pad, info, user_data):
        event = info.get_event()
        if event is None:
            return Gst.PadProbeReturn.PASS
        if info.get_event().type != Gst.EventType.RECONFIGURE:
            return Gst.PadProbeReturn.PASS
        log.info("dropping RECONFIGURE event from %s", info.get_event())
        return Gst.PadProbeReturn.DROP

    def add_input_branch(self, branch, identifier, volume=1.0):

        if not isinstance(identifier, str):
            identifier = str(identifier)

        pad = self.mixer.get_request_pad("sink_%u")
        pad.set_property("volume", volume)
        src_pad = branch.get_src_pad()

        sink_id = "sink_%s_%s" % (identifier, self.mixer_index)
        ghost_sink_pad = Gst.GhostPad.new(sink_id, pad)
        self.mixer_index += 1
        ghost_sink_pad.set_active(True)
        self.bin.add_pad(ghost_sink_pad)
        src_pad.link(ghost_sink_pad)

        input_info = AudioMixerInputInfo(
                ghost_sink_pad,
                pad,
                branch,
                [],
                identifier)
        self.identifier_to_input_info[identifier] = input_info

    def get_sink_pad(self, identifier):
        input_info = self.identifier_to_input_info[identifier]
        return input_info.mixer_sink_pad

    def add_output_branch(self, id, sink_pad):
        if not sink_pad.is_active():
            raise Exception("Pad Not Active: %s %s", id, sink_pad.get_name())
        ghost_src_pad = Gst.GhostPad.new_no_target(id, Gst.PadDirection.SRC)
        ghost_src_pad.set_active(True)
        self.bin.add_pad(ghost_src_pad)
        ghost_src_pad.link(sink_pad)
        tee_src_pad = self.tee.get_request_pad("src_%u")
        ghost_src_pad.set_target(tee_src_pad)

        output_info = AudioMixerOutputInfo(
                ghost_src_pad,
                tee_src_pad,
                sink_pad,
                id)

        self.identifier_to_output_info[id] = output_info

    def remove_output_branch(self, id):
        output_info = self.identifier_to_output_info.get(id)
        if not output_info:
            log.info("%s already removed", id)
            return
        ghost_src_pad = output_info.ghost_src_pad
        tee_src_pad = output_info.tee_src_pad

        self.get_bin().remove_pad(ghost_src_pad)
        self.tee.release_request_pad(tee_src_pad)

        del self.identifier_to_output_info[id]

    def remove_input_branch(self, identifier):
        input_info = self.identifier_to_input_info.get(identifier)
        if not input_info:
            log.debug("cleanup_branch - publisher was not linked %s", identifier)
            return
        mixer_sink_pad = input_info.mixer_sink_pad
        ghost_sink_pad = input_info.ghost_sink_pad
        self.get_bin().remove_pad(ghost_sink_pad)
        self.mixer.release_request_pad(mixer_sink_pad)
        del self.identifier_to_input_info[identifier]

    def destroy(self):
        if self.block_reconfigure_pad and self.block_probe_id:
            self.block_reconfigure_pad.remove_probe(self.block_probe_id)
        super(AudioMixer, self).destroy()

