import re
import gi
from gi.repository import Gst
from threading import RLock
from classes import logger as log
from classes import error
import time


# todo move to a meta class ;-)
_snake_re = re.compile('([A-Z])')
_names = set()

_name_lock = RLock()


def to_snake_case(given_value):
    return _snake_re.sub(r'_\1', given_value).lower().replace('__', '_').lstrip('_')


def get_branch_base_name(name):
    return re.sub(r'\d', "", name)


def get_element_base_name(name):
    return re.sub(r'\d', "", name)


def free_name(name):
    with _name_lock:
        _names.discard(name)


AUDIO_DEV_QUEUE_MAX_SIZE_TIME = 300000000  # 300 ms
AUDIO_MIXER_LATENCY = AUDIO_DEV_QUEUE_MAX_SIZE_TIME - 30000000   # - 30 ms


class Branch:

    def __init__(self, bus, session_id):
        self.bus = bus
        self.bin = None
        self.session_id = session_id
        self.type = to_snake_case(type(self).__name__)
        with _name_lock:
            self.name = self._mk_name()
            self.bin = Gst.Bin.new(self.name)
        log.info("creating branch %s %s", self, self.name)
        super(Branch, self).__init__()

    def draw_dot(self, name=None):
        if not name:
            name = f'{self.name}_{time.time()}'
        # if self.pipeline is None:
        #     log.error("No pipeline when asking for dot")
        #     return
        log.info("draw_dot: %s", name)
        Gst.debug_bin_to_dot_file(self.bin, Gst.DebugGraphDetails.ALL, name)

    def free_name(self):
        free_name(self.name)

    def _mk_name(self, i=0):
        global _names
        if i is 0:
            name = "%s" % to_snake_case(type(self).__name__)
        else:
            name = "%s%d" % (to_snake_case(type(self).__name__), i)
        if name in _names:
            i += 1
            return self._mk_name(i)
        _names.add(name)
        return name

    def make_queue(self, queue_name="", **kwargs):
        props = {k.replace("_", "-"): v for k, v in kwargs.items()}
        if queue_name:
            queue_name += "_"
        name = "%s_%squeue" % (self.name, queue_name)
        queue = Gst.ElementFactory.make("queue", name)
        queue.set_property("max-size-bytes", props.pop("max-size-bytes", 0))
        queue.set_property("max-size-buffers", props.pop("max-size-buffers", 0))
        queue.set_property("max-size-time", props.pop("max-size-time", 0))
        queue.set_property("silent", props.pop("silent", True))

        for k, v in props.items():
            queue.set_property(k, v)
        self.bin.add(queue)
        return queue

    def make(self, element_name, name=None, **kwargs):
        props = {k.replace("_", "-"): v for k, v in kwargs.items()}
        element = Gst.ElementFactory.make(element_name, name)
        if not element:
            raise TypeError('Failed to create element: {}, {} {}'.format(element_name, name, props))
        for k, v in props.items():
            element.set_property(k, v)
        self.bin.add(element)
        return element

    def make_caps(self, caps_str):
        return self.make("capsfilter", caps=Gst.Caps.from_string(caps_str))

    def get_bin(self):
        return self.bin

    # override this if not static pads
    def get_sink_pad(self):
        pad = self.bin.get_static_pad('sink')
        if not pad:
            log.error("sink pad not found %s", self)
            self.draw_dot(f'error_no_sink_pad_{self.name}_{int(time.time())}')
        return pad

    # override this if not static pads
    def get_src_pad(self):
        pad = self.bin.get_static_pad('src')
        return pad

    def release_request_pad(self, element, pad):
        structure = Gst.Structure.new_empty("release_request_pad")
        structure.set_value("element", element)
        structure.set_value("pad", pad)
        msg = Gst.Message.new_application(self.bin, structure)
        self.bus.post(msg)

    def signal_eos(self, identifier, bin):
        log.info("signal_eos %s" % identifier)
        structure = Gst.Structure.new_empty("eos")
        structure.set_value("identifier", identifier)
        msg = Gst.Message.new_application(self.bin, structure)
        self.bus.post(msg)

    def on_fatal_error(self, branch, err, status):
        return

    def make_default_queue(self, name):
        queue = self.make_queue(name, max_size_buffers=2, leaky=1)
        return queue

    def make_audio_device_queue(self, name="device"):
        queue = self.make_queue(name, max_size_time=AUDIO_DEV_QUEUE_MAX_SIZE_TIME, leaky=0)
        return queue

    def make_glupload_elements(self, **kwargs):
        pre_queue = self.make_default_queue("glupload_pre")
        glupload = self.make("glupload")
        post_queue = self.make_default_queue("glupload_post")
        elements = [pre_queue, glupload, post_queue]
        if kwargs.get("glcolorconvert", False):
            elements.append(self.make("glcolorconvert"))
        if kwargs.get("glcolorscale", False):
            scale = self.make("glcolorscale")
            width = kwargs.get("width", 0)
            height = kwargs.get("height", 0)
            capsfilter = self.make("capsfilter", caps=Gst.Caps.from_string("video/x-raw(memory:GLMemory),width={},height={},pixel-aspect-ratio=1/1".format(width, height)))
            elements.extend([scale, capsfilter])
        return elements

    def link_many(self, *args):
        if not len(args):
            return
        for index, element in enumerate(args[:-1]):
            next_element = args[index+1]
            log.debug("link_many linking %s to %s" %
                      (element.name, next_element.name))
            if not Gst.Element.link(element, next_element):
                log.error('{} failed to link {}, {}'.format(self.name, element, next_element))
                raise error.LinkError('{} failed to link {}, {}'.format(self.name, element, next_element))

    def cleanup(self):
        # this function is a lie
        log.error("CLEANUP")
        return

    # TODO: generalize the case of multple bins -- see VideoSrc
    def set_state(self, gst_state):
        return self.bin.set_state(gst_state)

    def destroy(self):
        log.info("destroying branch %s %s", self, self.name)
        self.bin.set_state(Gst.State.NULL)

