from .filter_branch import FilterBranch
from gi.repository import Gst, GObject


fragment_shader = """
#version 100
#ifdef GL_ES
precision mediump float;
#endif
varying vec2 v_texcoord;
uniform sampler2D tex;
uniform float time;
uniform float width;
uniform float height;
uniform float crop_left;
uniform float crop_right;
uniform float crop_bottom;
uniform float crop_top;

void main () {
    vec4 color;
    float x = v_texcoord.x;
    float y = v_texcoord.y;
    float cl_px = float(crop_left) / 100.0;
    float cr_px = 1.0 - float(crop_right) / 100.0;
    float ct_px = float(crop_top) / 100.0;
    float cb_px = 1.0 - float(crop_bottom) / 100.0;

    if(x > cl_px && x < cr_px && y > ct_px && y < cb_px) {
        gl_FragColor = texture2D(tex, v_texcoord);
    } else {
        gl_FragColor = vec4(0.0, 0.0, 0.0, 0.0);
    }
}
"""


class GLCrop(FilterBranch):

    def __init__(self, bus, session_id, fps):
        super(GLCrop, self).__init__(bus, session_id)
        self.fps = fps

        if self.has_mixer():
            self.setup_crop_with_mixer()
        else:
            self.setup_crop()
        self.link_many(*self.elements)
        self.add_static_ghost_pads()

    def has_mixer(self):
        return self.fps != 0

    def setup_crop_with_mixer(self):
        # This is a hack to make sure cropping works with static video sources such has static cef overlays
        # probably don't remove this unless you know of a better way. Videorate doesn't work because it only
        # decides to duplicate when it gets the next frame. Basically videorate only works well with non-live input
        identity = self.make("identity")
        videorate = self.make("glvideomixerelement", background="transparent")
        videorate.set_property("start-time-selection", 1)

        videorate_capsfilter = self.make("capsfilter")
        videorate_caps = Gst.Caps.from_string("video/x-raw(memory:GLMemory),framerate=%d/1" % self.fps)
        videorate_capsfilter.set_property("caps", videorate_caps)

        self.shader = self.make("glshader")
        self.shader.set_property("fragment", fragment_shader)

        self.elements = [identity, videorate, videorate_capsfilter, self.shader]

    def setup_crop(self):
        self.shader = self.make("glshader")
        self.shader.set_property("fragment", fragment_shader)
        self.elements = [self.shader]

    def update(self, position):
        crop_left_value = GObject.Value(GObject.TYPE_FLOAT, position["crop_left"])
        crop_right_value = GObject.Value(GObject.TYPE_FLOAT, position["crop_right"])
        crop_top_value = GObject.Value(GObject.TYPE_FLOAT, position["crop_top"])
        crop_bottom_value = GObject.Value(GObject.TYPE_FLOAT, position["crop_bottom"])

        uniform_structure = Gst.Structure.new_empty("uniform")
        uniform_structure.set_value("crop_left", crop_left_value)
        uniform_structure.set_value("crop_right", crop_right_value)
        uniform_structure.set_value("crop_top", crop_top_value)
        uniform_structure.set_value("crop_bottom", crop_bottom_value)
        self.shader.set_property("uniforms", uniform_structure)

