from .branch import Branch
from gi.repository import Gst
from classes import logger as log
from classes.util import link_many
from classes.stats.qos import add_cb, remove_cb, remove_stats


class VideoMixerInputInfo:

    def __init__(self, ghost_sink_pad, mixer_sink_pad, branch, elements, identifier):
        self.ghost_sink_pad = ghost_sink_pad
        self.mixer_sink_pad = mixer_sink_pad
        self.branch = branch
        self.identifier = identifier
        self.elements = elements


class VideoMixerOutputInfo:
    def __init__(self, ghost_src_pad, tee_src_pad, sink_pad, identifier):
        self.ghost_src_pad = ghost_src_pad
        self.tee_src_pad = tee_src_pad
        self.identifier = identifier
        self.sink_pad = sink_pad


class GLVideoMixer(Branch):

    def __init__(self, bus, session_id, width, height, fps, is_dxgi):
        super(GLVideoMixer, self).__init__(bus, session_id)
        self.mixer_index = 0
        self.is_dxgi = is_dxgi

        self.identifier_to_input_info = {}
        self.identifier_to_output_info = {}

        self.width = width
        self.height = height
        self.fps = fps
        video_mixer = self.make("glvideomixerelement", background="black")
        caps = Gst.Caps.from_string("video/x-raw(memory:GLMemory),width=%d,height=%d,framerate=%d/1" % (self.width, self.height, self.fps))
        compositor_capsfilter = self.make("capsfilter", caps=caps)
        self.mixer = video_mixer
        mixer_queue = self.make_queue("mixer", max_size_buffers=2, leaky=1)

        elements = [video_mixer,
                    compositor_capsfilter,
                    mixer_queue]

        if self.is_dxgi:
            self.gl2dxgi = self.make("gl2dxgi")
            elements.append(self.gl2dxgi)

        self.tee = self.make("tee")
        self.tee.set_property("allow-not-linked", True)
        elements.append(self.tee)

        link_many(*elements)

        before_tee = elements[-2]
        self.block_reconfigure_pad = before_tee.get_static_pad("src")
        self.block_probe_id = self.block_reconfigure_pad.add_probe(
            Gst.PadProbeType.EVENT_UPSTREAM | Gst.PadProbeType.BLOCK,
            self.reconfigure_cb,
            None)

        self.start_stats()

    def set_property(self, prop_name, value):
        return self.mixer.set_property(prop_name, value)

    def reconfigure_cb(self, pad, info, user_data):
        event = info.get_event()
        if event is None:
            return Gst.PadProbeReturn.PASS
        if info.get_event().type != Gst.EventType.RECONFIGURE:
            return Gst.PadProbeReturn.PASS
        log.info("dropping RECONFIGURE event from %s", info.get_event())
        return Gst.PadProbeReturn.DROP

    def add_input_branch(self, branch, identifier, pixel_position, z_order, alpha):
        if not isinstance(identifier, str):
            identifier = str(identifier)

        if identifier in self.identifier_to_input_info:
            log.error("already have identifier %s in video_mixer", identifier)

        src_pad = branch.get_src_pad()

        try:
            sink_id = "sink_%s_%s" % (identifier, self.mixer_index)
            self.mixer_index += 1

            ghost_sink_pad = Gst.GhostPad.new_no_target(
                sink_id, Gst.PadDirection.SINK)
            ghost_sink_pad.set_active(True)
            self.bin.add_pad(ghost_sink_pad)
            src_pad.link(ghost_sink_pad)

            pad = self.mixer.get_request_pad("sink_%u")
            pad.set_property("xpos", pixel_position['x'])
            pad.set_property("ypos", pixel_position['y'])
            pad.set_property("width", pixel_position['width'])
            pad.set_property("height", pixel_position['height'])
            pad.set_property("zorder", z_order)
            pad.set_property("alpha", alpha)
            ghost_sink_pad.set_target(pad)

            input_info = VideoMixerInputInfo(
                ghost_sink_pad,
                pad,
                branch,
                [],
                identifier)

            self.identifier_to_input_info[identifier] = input_info
        except Gst.LinkError as linkError:
            log.error("LinkError %s id: %s", linkError, identifier)

    def update_input_branch(self, identifier, pixel_position, z_order, alpha):

        if not isinstance(identifier, str):
            identifier = str(identifier)

        input_info = self.identifier_to_input_info.get(identifier)
        if input_info is None:
            return
        mixer_sink_pad = input_info.mixer_sink_pad
        mixer_sink_pad.set_property("xpos", pixel_position['x'])
        mixer_sink_pad.set_property("ypos", pixel_position['y'])
        mixer_sink_pad.set_property("width", pixel_position['width'])
        mixer_sink_pad.set_property("height", pixel_position['height'])
        mixer_sink_pad.set_property("zorder", z_order)
        mixer_sink_pad.set_property("alpha", alpha)

    def remove_input_branch(self, identifier):
        input_info = self.identifier_to_input_info.get(identifier)
        if not input_info:
            log.debug("cleanup_branch - publisher was not linked %s", identifier)
            return
        mixer_sink_pad = input_info.mixer_sink_pad
        ghost_sink_pad = input_info.ghost_sink_pad
        self.get_bin().remove_pad(ghost_sink_pad)
        self.mixer.release_request_pad(mixer_sink_pad)
        del self.identifier_to_input_info[identifier]

    def get_sink_pad(self, identifier):
        input_info = self.identifier_to_input_info[identifier]
        return input_info.mixer_sink_pad

    def add_output_branch(self, id, sink_pad):
        if not sink_pad.is_active():
            raise Exception("Pad Not Active: %s %s", id, sink_pad.get_name())

        try:
            ghost_src_pad = Gst.GhostPad.new_no_target(
                id, Gst.PadDirection.SRC)
            ghost_src_pad.set_active(True)
            self.bin.add_pad(ghost_src_pad)
            ghost_src_pad.link(sink_pad)
            tee_src_pad = self.tee.get_request_pad("src_%u")
            ghost_src_pad.set_target(tee_src_pad)

            output_info = VideoMixerOutputInfo(
                ghost_src_pad,
                tee_src_pad,
                sink_pad,
                id)
            self.identifier_to_output_info[id] = output_info
        except Gst.LinkError as linkError:
          log.error("LinkError %s srcpad: %s  sinkpad: %s idno need to cleanup branch: %s",
                    linkError, ghost_src_pad.get_name, sink_pad.get_name, id)

    def remove_output_branch(self, id):
        output_info = self.identifier_to_output_info.get(id)
        if not output_info:
            log.info("%s already removed", id)
            return
        ghost_src_pad = output_info.ghost_src_pad
        tee_src_pad = output_info.tee_src_pad

        self.get_bin().remove_pad(ghost_src_pad)
        self.tee.release_request_pad(tee_src_pad)

        del self.identifier_to_output_info[id]

    def start_stats(self):
        if self.is_dxgi:
            add_cb(self.add_stats)

    def end_stats(self):
        if self.is_dxgi:
            remove_cb(self.add_stats)
            remove_stats(["dxgi_latency_nr",
                          "dxgi_max_latency_nr",
                          "dxgi_delay_nr",
                          "dxgi_max_delay_nr"])

    def add_stats(self):
        return {
                "dxgi_latency_nr": self.gl2dxgi.get_property("latency"),
                "dxgi_max_latency_nr": self.gl2dxgi.get_property("max-latency"),
                "dxgi_delay_nr": self.gl2dxgi.get_property("delay"),
                "dxgi_max_delay_nr": self.gl2dxgi.get_property("max-delay")
                }

    def destroy(self):
        if self.block_reconfigure_pad and self.block_probe_id:
            self.block_reconfigure_pad.remove_probe(self.block_probe_id)
        super(GLVideoMixer, self).destroy()

