import time
import os
import sys
PYTHON_DIR = os.path.split(sys.executable)[0]
BASE_DIR = os.path.split(PYTHON_DIR)[0]
BEBO_DLLS_DIR_DEFAULT = os.path.join(BASE_DIR, 'bebodlls')
BEBO_DLLS_DIR = BEBO_DLLS_DIR_DEFAULT

from .branch import Branch
from gi.repository import Gst
from classes import logger as log
from classes.stats.qos import add_cb, remove_cb, remove_stats
from classes.ratecontrol import BitRateControl
from classes.util import link_many

MIN_FPS = 10
MIN_BITRATE = 1000

# GstX264EncPreset
GST_X264_PRESETS = {
  0: None,
  1: "ultrafast",
  2: "superfast",
  3: "veryfast",
  4: "faster",
  5: "fast",
  6: "medium",
  7: "slow",
  8: "slower",
  9: "veryslow",
  10: "placebo"
}


def gst_x264_enc_preset_to_string(preset):
    try:
        p = int(preset)
        out = GST_X264_PRESETS.get(p, preset)
        if out is None:
            log.error("Invalid x264 preset %s", preset)
        preset = out
    except ValueError:
        pass

    if preset is None:
        log.info("defaulting x264 preset to ultrafast")
        preset = "ultrafast"
    return preset


class Rtmp(Branch, BitRateControl):

    def __init__(self, bus, session_id, width, height, fps, encoder, encoder_preset, bitrate,
                 location, session_setting):
        super(Rtmp, self).__init__(bus, session_id)
        self.session_setting = session_setting
        self.width = width
        self.height = height
        self.fps = fps
        self.target_fps = fps
        self.min_fps = max(int(self.fps / 2), MIN_FPS)
        self.bitrate = bitrate
        self.target_bitrate = bitrate
        self.min_bitrate = max(int(self.bitrate * 0.33), MIN_BITRATE)
        self.fps_last_up = 0
        self.encoder = encoder
        self.video_encoder = None
        self.videorate_capsfilter = None
        self.video_encoder_capsfilter = None
        log.info("streaming bitrate %d at location %s", bitrate, location)

        # Video Encoder
        # video_elements = self.make_video_source_elements()
        # video_elements.extend(self.make_video_encoder_elements(encoder, encoder_preset, width, height))
        video_elements = self.make_video_encoder_elements(encoder, encoder_preset, width, height)
        h264parse = self.make("h264parse")
        video_elements.append(h264parse)
        link_many(*video_elements)

        # Audio Encoder
        # audio_elements = self.make_audio_source_elements()
        # audio_elements.extend(self.make_audio_encoder_elements())
        audio_elements = self.make_audio_encoder_elements()
        aacparse = self.make("aacparse")
        audio_elements.append(aacparse)
        link_many(*audio_elements)

        # flv -> rtmp
        flvmux = self.make("flvmux", streamable=True)
        self.rtmp_queue = self.make_queue("rtmp", max_size_time=2000000000)
        self.rtmp_sink = self.make("rtmp2sink", location=location)
        error_ignore = self.make("errorignore", ignore_notnegotiated=False)

        Gst.Element.link(video_elements[-1], flvmux)
        Gst.Element.link(audio_elements[-1], flvmux)

        link_many(flvmux, self.rtmp_queue, error_ignore, self.rtmp_sink)

        video_sink_pad = video_elements[0].get_static_pad("sink")
        if video_sink_pad:
            self.ghost_video_sink_pad = Gst.GhostPad.new("video_sink", video_sink_pad)
            self.bin.add_pad(self.ghost_video_sink_pad)
        audio_sink_pad = audio_elements[0].get_static_pad("sink")
        if audio_sink_pad:
            self.ghost_audio_sink_pad = Gst.GhostPad.new("audio_sink", audio_sink_pad)
            self.bin.add_pad(self.ghost_audio_sink_pad)
        self.set_bitrate()
        self.start_stats()

    def make_video_source_elements(self):
        gamecapture = self.make('gamecapture',
                window_name="Fortnite  ",
                class_name="UnrealWindow",
                inject_dll_path=BEBO_DLLS_DIR,
                anti_cheat=True)
        gamecapture_caps = self.make_caps(
                "video/x-raw(memory:GLMemory),format=RGBA,width=%d,height=%d,framerate=%d/1" %
                (self.width, self.height, self.fps))
        gamecapture_queue = self.make_queue("gamecapture")
        return [gamecapture, gamecapture_caps, gamecapture_queue]

    def make_audio_source_elements(self):
        try:
            if self.session_setting.enable_desktop_audio:
                audiosrc = self.make('audiotestsrc',
                        wave='silence',
                        is_live=True,
                        do_timestamp=True)
            else:
                audiosrc = self.make('wasapisrc',
                        loopback=True,
                        do_timestamp=False,
                        low_latency=True,
                        provide_clock=False,
                        slave_method=0)
        except Exception as e:
            log.error('Failed to create desktop audio, creating a silence test src instead')
            log.exception(e)
            audiosrc = self.make('audiotestsrc',
                    wave='silence',
                    is_live=True,
                    do_timestamp=True)
        audio_queue = self.make_audio_device_queue('desktopaudio')
        audioconvert = self.make("audioconvert")
        audioresample = self.make("audioresample")
        return [audiosrc, audio_queue, audioconvert, audioresample]

    def make_video_encoder_elements(self, encoder, encoder_preset, width, height):
        elements = []
        if encoder == 'nvenc':
            log.info("streaming with nvenc")
            self.codec_implementation_tx = "nvh264enc"
            self.codec_preset = "hq"
            elements.extend(self.make_nvenc_elements(encoder_preset, width, height))
        else:
            self.codec_implementation_tx = "x264enc"
            self.codec_preset = gst_x264_enc_preset_to_string(encoder_preset)
            log.info("streaming with x264 preset: {}".format(self.codec_preset))
            elements.extend(self.make_x264_elements(encoder_preset, width, height))
        post_queue = self.make_queue("post_video_enc",
                max_size_time=200000000,
                leaky=0)
        elements.append(post_queue)
        return elements

    def make_audio_encoder_elements(self):
        pre_audio_encoder_queue = self.make_queue("audio_encoder",
            max_size_time=3000000000,
            leaky=1)
        audio_encoder = self.make("avenc_aac", bitrate=128000)
        post_audio_encoder_queue = self.make_queue("post_audio_enc",
                max_size_time=6000000000,
                leaky=0)
        audio_encoder_capsfilter = self.make("capsfilter",
                caps=Gst.Caps.from_string("audio/mpeg"))
        return [pre_audio_encoder_queue, audio_encoder, post_audio_encoder_queue,
                audio_encoder_capsfilter]

    def make_nvenc_elements(self, encoder_preset, width, height):
        video_enc_queue = self.make_queue("video_encoder",
                                          silent=False,
                                          max_size_buffers=2,
                                          leaky=1)
        self.video_encoder = self.make('d3dnvh264enc')
        self.video_encoder.set_property("rc-mode", "cbrhq")
        self.video_encoder.set_property("preset", "hq")
        self.video_encoder.set_property("gop-size", self.target_fps * 10)
        self.video_encoder_capsfilter = self.make_caps("video/x-h264,width=%d,height=%d,framerate=%d/1" % (width, height, self.target_fps))
        return [video_enc_queue, self.video_encoder,
                self.video_encoder_capsfilter]

    def make_x264_elements(self, encoder_preset, width, height):
        colorconvert = self.make('glcolorconvert')
        colorconvert_capsfilter = self.make('capsfilter', caps=Gst.Caps.from_string('video/x-raw(memory:GLMemory),format=NV12'))
        pre_dl_queue = self.make_default_queue("pre_dl_queue")
        gldownload = self.make('gldownload')
        post_dl_queue = self.make_default_queue("post_dl_queue")
        video_enc_queue = self.make_queue("video_encoder",
                                          silent=False,
                                          max_size_time=200000000,
                                          leaky=1)
        self.codec_preset = gst_x264_enc_preset_to_string(encoder_preset)
        self.video_encoder = self.make("x264enc")
        self.video_encoder.set_property("speed-preset", self.codec_preset)
        self.video_encoder.set_property("key-int-max", self.target_fps * 2)
        self.video_encoder.set_property("pass", "cbr")
        self.video_encoder.set_property("vbv-buf-capacity", 2000)
        self.video_encoder.set_property("option-string", "filler=1:scenecut=0")
        caps = Gst.Caps.from_string("video/x-h264,width=%d,height=%d,framerate=%d/1" % (width, height, self.target_fps))
        self.video_encoder_capsfilter = self.make('capsfilter', caps=caps)
        return [colorconvert, colorconvert_capsfilter, pre_dl_queue, gldownload, post_dl_queue,
                video_enc_queue, self.video_encoder, self.video_encoder_capsfilter]

    def get_error_message(self, src, error, stream_id):
        code = error.code
        msg = {"url": "/stream/error",
               "stream_id": stream_id,
               "message": "%s" % error,
               "code": 500}

        if code == 1:
            # GST_RESOURCE_ERROR_FAILED
            msg['code'] = 499
            msg['message'] = "Failed to connect"
        elif code == 3:
            # GST_RESOURCE_ERROR_NOT_FOUND
            msg['code'] = 404
            msg['message'] = "Not Found"
        elif code == 6:
            # GST_RESOURCE_ERROR_OPEN_WRITE,
            msg['code'] = 499
            msg['message'] = "Could not open resource"
        elif code == 10:
            # GST_RESOURCE_ERROR_WRITE,
            msg['code'] = 408
            msg['message'] = "Connection Dropped"
        elif code == 15:
            # GST_RESOURCE_ERROR_NOT_AUTHORIZED,
            # only rtmp2sink
            msg['code'] = 401
            msg['message'] = "Not Authorized"
        return msg

    def video_queue_overrun(self, queue):
        fps = max(self.target_fps - 5, self.min_fps)
        if fps != self.target_fps:
            log.info("video enc queue overrun settings fps %d -> %d", self.target_fps, fps)
            self.videorate_capsfilter.set_property("caps", Gst.Caps.from_string("framerate=%d/1" % fps))
            if self.encoder == "nvenc":
                self.video_encoder.set_property("gop-size", self.target_fps * 2)
            else:
                self.video_encoder.set_property("key-int-max", self.target_fps * 2)
            self.target_fps = fps

    def get_time(self):
        return int(time.perf_counter() * 1000)

    def video_queue_underrun(self, queue):
        if self.get_time() < (self.fps_last_up + 2000):
            return
        self.fps_last_up = self.get_time()

        fps = min(self.target_fps + 1, self.fps)
        if fps != self.target_fps:
            log.info("video enc queue underrun settings fps %d -> %d", self.target_fps, fps)
            self.videorate_capsfilter.set_property("caps", Gst.Caps.from_string("framerate=%d/1" % fps))
            if self.encoder == "nvenc":
                self.video_encoder.set_property("gop-size", self.target_fps * 2)
            else:
                self.video_encoder.set_property("key-int-max", self.target_fps * 2)
            self.target_fps = fps

    def get_video_sink_pad(self):
        return self.ghost_video_sink_pad

    def get_audio_sink_pad(self):
        return self.ghost_audio_sink_pad

    def set_bitrate(self):
        self.video_encoder.set_property("bitrate", self.target_bitrate)

    def reset_bitrate(self, bitrate):
        if bitrate == "default":
            log.error("invalid bitrate", bitrate)
            return
        self.bitrate = bitrate
        self.target_bitrate = bitrate
        self.set_bitrate()

    def update_bitrate(self, bitrate):
        if bitrate and self.bitrate != bitrate:
            log.info("setting bitrate was %s now %s" % (self.bitrate, bitrate))
            self.reset_bitrate(bitrate)

    def destroy(self):
        super(Rtmp, self).destroy()
        self.rtmp_queue = None
        self.bwe_destroy()
        self.rtmp_sink = None
        self.end_stats()

    def start_stats(self):
        add_cb(self.add_stats)

    def end_stats(self):
        remove_cb(self.add_stats)
        remove_stats(["fps_nr",
                      "width_nr",
                      "height_nr",
                      "target_fps_nr",
                      "target_bitrate_nr",
                      "estimated_bitrate_nr",
                      "bitrate_nr",
                      "codec_preset_tx",
                      "codec_implementation_tx"])

    def add_stats(self):
        stats = {"fps_nr": self.fps,
                 "width_nr": self.width,
                 "height_nr": self.height,
                 "target_fps_nr": self.target_fps,
                 "target_bitrate_nr": self.target_bitrate,
                 "estimated_bitrate_nr": self.get_estimated_upper_bound(),
                 "bitrate_nr": self.bitrate,
                 "codec_preset_tx": self.codec_preset,
                 "codec_implementation_tx": self.codec_implementation_tx}
        if self.encoder == "nvenc" and self.video_encoder:
            stats["current_fps_nr"] = self.video_encoder.get_property("fps")
        if self.rtmp_sink:
            stats["current_bitrate_nr"] = self.rtmp_sink.get_property("bitrate")
        return stats

