import os
import time
from .src_branch import LinearSrcBranch
from gi.repository import Gst
from classes import logger as log
from classes.util import link_many, clamp
import threading

DEFAULT_LOOPBACK_VOLUME = 0.7


class VideoSrc(LinearSrcBranch):

    def __init__(self, bus, session_id, location, on_finished=None, **kwargs):

        super(VideoSrc, self).__init__(bus, session_id)

        log.info("VideoSrc name: %s, adding: %s, %s", self.name, location, kwargs)

        self.init(location, **kwargs)
        video_channel = self.create_video_channel_name(location)
        audio_channel = self.create_audio_channel_name(location)

        self.create_interpipeline()
        self.lock_interpipeline()
        self.create_filesrc_and_decodebin(location)
        self.on_finished = on_finished

        video_player_elements = self.create_video_player(video_channel, **kwargs)
        self.create_intervideosrc(video_channel, **kwargs)

        audio_player_elements = self.create_audio_player(audio_channel)

        audio_sink = audio_player_elements[0] if len(audio_player_elements) > 0 else None
        video_sink = video_player_elements[0]
        self.intervideosink = video_player_elements[-1]

        self.decodebin.connect("pad-added", self.player_ready_and_unlock, video_sink, audio_sink)

        self.interpipeline.set_state(Gst.State.PAUSED)

        self.wait_for_player_ready()
        self.remove_stale_audio_elements()

    def init(self, location, **kwargs):
        self.location = location
        self.audio_connected = False
        self.video_connected = False
        self.volume_element = None

        self.loop = kwargs.get('loop', False)
        self.audio_output_type = kwargs.get('audio_output_type', None)

    def create_interpipeline(self):
        self.interpipeline = Gst.Pipeline.new("inter")
        self.interpipeline_bus = self.interpipeline.get_bus()
        self.interpipeline_bus.add_signal_watch()
        self.interpipeline_bus.connect("message", self.interpipeline_on_message)

    def lock_interpipeline(self):
        self.video_lock = threading.Lock()
        self.video_lock.acquire()
        self.audio_lock = threading.Lock()
        self.audio_lock.acquire()

    def create_filesrc_and_decodebin(self, location):
        self.decodebin = Gst.ElementFactory.make("decodebin")

        filesrc = Gst.ElementFactory.make("filesrc")
        filesrc.set_property("location", location)

        self.interpipeline.add(filesrc)
        self.interpipeline.add(self.decodebin)
        link_many(filesrc, self.decodebin)

    def create_video_channel_name(self, location):
        ts = int(time.time())
        return 'video_{}_{}'.format(location, ts)

    def create_audio_channel_name(self, location):
        ts = int(time.time())
        return 'audio_{}_{}'.format(location, ts)

    def create_video_player(self, channel_name, **kwargs):
        self.deinterlace = Gst.ElementFactory.make("deinterlace")
        forceprogressive = Gst.ElementFactory.make("capsfilter")
        forceprogressive.set_property("caps",
                                      Gst.Caps.from_string("video/x-raw,interlace-mode=(string)progressive"))

        player_elements = [self.deinterlace, forceprogressive]

        if kwargs.get('include_alpha', False):
            alpha = Gst.ElementFactory.make("alpha")
            alpha.set_property("method", 3)
            alpha.set_property("target-g", 255)
            alpha.set_property("white-sensitivity", 128)
            alpha.set_property("black-sensitivity", 128)
            alpha.set_property("noise-level", 0)
            alpha.set_property("angle", 90)
            player_elements.append(alpha)

        intervideosink = Gst.ElementFactory.make("intervideosink")
        intervideosink.set_property('channel', channel_name)
        player_elements.append(intervideosink)
        for e in player_elements:
            self.interpipeline.add(e)

        link_many(*player_elements)
        return player_elements

    def create_intervideosrc(self, channel_name, **kwargs):
        intervideosrc = self.make("intervideosrc")
        intervideosrc.set_property('channel', channel_name)
        intervideosrc.set_property('timeout', 100000000)
        self.elements = [intervideosrc]

        # we need to downscale it (to canvas size) here otherwise we'd try to load in 4k / 8k video into the pipeline.
        self.elements.extend(self.make_glupload_elements(glcolorconvert=True, glcolorscale=True, width=kwargs.get('width', 1280), height=kwargs.get('height', 720)))
        link_many(*self.elements)
        self.add_static_ghost_pad()

    def create_audio_player(self, channel_name):
        self.inter_audiosinks = []
        if self.audio_output_type == 'loopback':
            audioconvert = Gst.ElementFactory.make("audioconvert")
            audioresample = Gst.ElementFactory.make("audioresample")
            audiosink = Gst.ElementFactory.make("wasapisink")
            audiosink.set_property("use-audioclient3", False)
            self.volume_element = Gst.ElementFactory.make("volume")
            self.volume_element.set_property("volume", DEFAULT_LOOPBACK_VOLUME)
            self.inter_audiosinks = [audioconvert, audioresample, self.volume_element, audiosink]
        elif self.audio_output_type == 'mixer': # NOT WORKING YET
            interaudiosink = Gst.ElementFactory.make("interaudiosink")
            interaudiosink.set_property('channel', channel_name)
            interaudiosrc = self.make("interaudiosrc")
            interaudiosrc.set_property('channel', channel_name)
            self.inter_audiosinks = [interaudiosink]
        else:
            self.audio_lock.release()

        # add interpipeline audio to the pipeline and link them
        if len(self.inter_audiosinks) > 0:
            for audio_element in self.inter_audiosinks:
                self.interpipeline.add(audio_element)
            link_many(*self.inter_audiosinks)
        return self.inter_audiosinks

    def wait_for_player_ready(self):
        # In mercy, branches / publishers are setup with the assumption that sources pad is always STATIC (always there).
        # For example, after a branch is created, it'll try to link the pads immediately.
        # But, for video files, the audio and video pads are SOMETIMES pad, where it'll need to parse/demux,
        # then tell us if the audio or video pad is available asynchronously using pad-added event.
        #
        # Ideally, we'd want to make mercy supports pad link asynchronously? but it'll probably requires a decent amount of work
        # So, I left this part as it was before (how it was setup for transitions)
        # These locks are in place, so that it'd keep the pads always available after this branch is constructed.
        self.video_lock.acquire(timeout=1)
        log.debug("video lock, video_connected: %d", self.video_connected)

        self.audio_lock.acquire(timeout=1)
        log.debug("audio lock, audio_connected: %d", self.audio_connected)

    def remove_stale_audio_elements(self):
        # if we dont remove the audio elements: mp4 files that don't have audio stream will cause the video to freeze.
        if not self.audio_connected:
            self.volume_element = None
            for audio_element in self.inter_audiosinks:
                self.interpipeline.remove(audio_element)

    def draw_dots(self):
        name = "interdot_%d" % time.time()
        log.info("drawing dot to %s\\%s.dot", os.environ['GST_DEBUG_DUMP_DOT_DIR'], name )
        Gst.debug_bin_to_dot_file(self.interpipeline, Gst.DebugGraphDetails.ALL, name);

    def set_loopback_volume(self, volume):
        if self.audio_connected and self.volume_element:
            clamped_volume = clamp(volume, 0, 200) # min volume: 0% ~ max volume: 200%
            self.volume_element.set_property("volume", round(clamped_volume / 100, 2))

    def set_loopback_mute(self, mute):
        if self.audio_connected and self.volume_element:
            self.volume_element.set_property("mute", mute)

    def destroy(self):
        super(VideoSrc, self).destroy()
        self.interpipeline.set_state(Gst.State.NULL)
        self.interpipeline = None
        self.location = None
        self.volume_element = None
        self.loop = None
        self.interpipeline_bus = None
        self.audio_lock = None
        self.video_lock = None
        self.audio_connected = None
        self.video_connected = None
        self.inter_audiosinks = None
        self.audio_output_type = None
        self.bin = None

    def player_ready_and_unlock(self, el, pad, videosink, audiosink):
        caps = pad.get_current_caps().to_string()
        log.info("pad_added name: %s file: %s caps: %s", self.name, self.location, caps)
        if "video" in caps and videosink:
            el.link(videosink)
            self.video_lock.release()
            self.video_connected = True
        if "audio" in caps and audiosink:
            el.link(audiosink)
            self.audio_lock.release()
            self.audio_connected = True
        # self.draw_dots()

    def set_state(self, gst_state):
        self.interpipeline.set_state(gst_state)
        return self.bin.set_state(gst_state)

    def seek_to_start(self):
        self.interpipeline.send_event(
            Gst.Event.new_seek(1.0,
                               Gst.Format.TIME,
                               Gst.SeekFlags.FLUSH | Gst.SeekFlags.SEGMENT,
                               Gst.SeekType.SET, 0,
                               Gst.SeekType.NONE, 0))

    def interpipeline_on_message(self, bus, message):
        t = message.type
        src = message.src

        if t == Gst.MessageType.WARNING:
            (error, info) = message.parse_warning()
            error_string = "%s" % error
            info_string = "%s" % info
            log.error("interpipeline: warning: %s, info:%s", error_string, info_string)
        elif t == Gst.MessageType.ERROR:
            (error, info) = message.parse_error()
            error_string = "%s" % error
            info_string = "%s" % info
            log.error("interpipeline: warning: %s, info:%s", error_string, info_string)
        elif t == Gst.MessageType.EOS or t == Gst.MessageType.SEGMENT_DONE:
            log.info("END OF VIDEO INTER VIDEO %s %s", t, src)
            if self.loop:
                self.seek_to_start()
            #     if self.on_finished:
            #         self.on_finished()
            #     return
        elif t == Gst.MessageType.STATE_CHANGED:
            (oldstate, newstate, pending) = message.parse_state_changed()
            log.debug("INTER PIPELINE STATE_CHANGED src: %s oldstate: %s newstate: %s, pending: %s", src.name, oldstate, newstate, pending)
        else:
            log.debug('interpipeline: Unknown message: {}, {}, {}'.format(t, src, str(src)))
            # pass
        return True

