from .src_branch import LinearSrcBranch
from gi.repository import Gst
from classes import logger as log
from threading import Timer
from .branch import AUDIO_MIXER_LATENCY


def generate_caps(channels, samplerate):
    caps_string = "audio/x-raw"

    if channels:
        caps_string = caps_string + ",channels=(int){}".format(channels)

    if samplerate:
        caps_string = caps_string + ",rate=(int){}".format(samplerate)

    return Gst.Caps.from_string(caps_string)


class Wasapi(LinearSrcBranch):

    def __init__(self, bus, session_id, device, is_mono, refresh_session_cb):

        super(Wasapi, self).__init__(bus, session_id)
        self.src = None
        self.msg_handle = None
        self.restart_required = False
        self.timer_handle = Timer(120, self.monitor)
        self.timer_handle.start()
        self.refresh_session_cb = refresh_session_cb
        log.info("Wasapi created - %s device_path: %s", self.name, device)
        self.device = device
        self.is_mono = is_mono
        src = self.make("wasapisrc",
                        do_timestamp=False,
                        low_latency=True,
                        provide_clock=False,
                        slave_method=0)
        self.src = src
        if device == "loopback":
            src.set_property("loopback", True)
        else:
            src.set_property("device", device)

        # the right hand side (audiomixer) is going to provide the caps that it needs and
        # audioconvert & audioresample will do the job if necessary
        audioconvert = self.make("audioconvert")
        audioresample = self.make("audioresample")

        # 200ms, cause directsound has 200ms latency, no idea why tho.
        # otherwise, we'd get fail to configure latency errora
        # June 27th: We've since changed our directsound plugin, we should remeasure the latency of the plugin.
        self.audio_queue = self.make_audio_device_queue()

        if is_mono:
            channel_mix = self.make('audiochannelmix')
            channel_mix.set_property('left-to-right', 1)
            audioconvert2 = self.make('audioconvert')
            self.elements = [src, audioconvert, channel_mix,
                             audioconvert2, audioresample, self.audio_queue]
        else:
            self.elements = [src, audioconvert,
                             audioresample, self.audio_queue]
        self.link_many(*self.elements)
        self.add_static_ghost_pad()
        self.msg_handle = self.bus.connect("message::state-changed",
                                           self.on_state_changed)


    def monitor(self):
        if not self.audio_queue:
            return
        try:
            self.timer_handle = Timer(120, self.monitor)
            self.timer_handle.start()
            # get_property("current-level-time") gives nanosec 150000000ns = 150ms
            if self.audio_queue.get_property("current-level-time") > AUDIO_MIXER_LATENCY:
                log.warning("Queue Size got too big %s -> Refreshing Session",
                            self.audio_queue.get_property("current-level-time"))
                self.restart_required = True
                self.refresh_session_cb()

        except Exception:
            log.exception("Unknown Execption in monitor")

    def log_real_name(self):
        rate = self.src.get_property("sample-rate")
        description = self.src.get_property("description")
        log.info("Wasapi device info - %s device_name: %s, device_path: %s, samplerate: %d, is_mono: %d",
                 self.name, description, self.device, rate, self.is_mono)

    def on_state_changed(self, bus, msg):
        old, new, pending = msg.parse_state_changed()
        if not msg.src == self.src:
            return

        if new == Gst.State.PLAYING:
            t = Timer(1.0, self.log_real_name)
            t.start()
            if self.msg_handle:
                self.bus.handler_disconnect(self.msg_handle)

    def on_message(self, bus, message):
        log.info("%s %s %s", self.name, bus, message)
        return

    def need_update(self):
        # return self.src and self.src.get_property('restart-required')
        return self.src and (self.src.get_property('restart-required') or self.restart_required)

    def destroy(self):
        super(Wasapi, self).destroy()
        if self.timer_handle:
            self.timer_handle.cancel()  # FIXME for some reason doesn't cancel
            self.timer_handle = None
        if self.msg_handle:
            self.bus.handler_disconnect(self.msg_handle)

