import json
import math

from classes import logger as log
from classes.util import sha1_hexdigest, index_of
from device_provider import device_provider
# NOTE: go to commit: bb29d146e5979faa534fb076753ef300dd524492
# to find a working device_factory
from gi.repository import Gst

PREFERRED_FORMAT = ["I420", "YV12", "NV12", "NV21", "UYVY", "YUY2", \
                    "MJPEG", "H264", \
                    "RGBA", "RGBx", "BGRA", "BGRx", "RGB", "BGR"]


class Device(dict):
    def __init__(self, **kwargs):
        super(Device, self).__init__(**kwargs)
        self.__dict__ = self


def get_video_device(request_device_id):
    return _find_device(_get_video_devices(), request_device_id)


def get_webcam_device_id(request_device_id):
    device = _find_device(_get_webcam_devices(), request_device_id)
    if device:
        return device['device_id']


def get_capturecard_device_id(request_device_id):
    device = _find_device(_get_capturecard_devices(), request_device_id)
    if device:
        return device['device_id']


def get_audio_device(request_device_id):
    return _find_device(_get_audio_devices(), request_device_id)


def is_best_match_caps(lhs_cap, rhs_cap, requested_cap):
    diff_height_lhs = abs(lhs_cap.get('height') - requested_cap.get('height'))
    diff_height_rhs = abs(rhs_cap.get('height') - requested_cap.get('height'))
    if diff_height_lhs != diff_height_rhs:
        return diff_height_lhs < diff_height_rhs

    diff_width_lhs = abs(lhs_cap.get('width') - requested_cap.get('width'))
    diff_width_rhs = abs(rhs_cap.get('width') - requested_cap.get('width'))
    if diff_width_lhs != diff_width_rhs:
        return diff_width_lhs < diff_width_rhs

    diff_fps_lhs = math.fabs(lhs_cap.get('framerate') - requested_cap.get('framerate'))
    diff_fps_rhs = math.fabs(rhs_cap.get('framerate') - requested_cap.get('framerate'))
    if diff_fps_lhs != diff_fps_rhs:
        return diff_fps_lhs < diff_fps_rhs

    # if output type is not found in preferred, it's probably not what we want
    lhs_cap_format = index_of(PREFERRED_FORMAT, lhs_cap.get('pixel_format'), len(PREFERRED_FORMAT) + 1)
    rhs_cap_format = index_of(PREFERRED_FORMAT, rhs_cap.get('pixel_format'), len(PREFERRED_FORMAT) + 1)
    if lhs_cap_format != rhs_cap_format:
        return lhs_cap_format < rhs_cap_format

    return False


def get_preferred_caps(device, request_preferred_format):
    video_device_caps = device.caps
    if len(video_device_caps) == 0:
        return None

    best_cap = None
    for cap in video_device_caps:
        if best_cap is None:
            best_cap = cap
            continue

        if is_best_match_caps(cap, best_cap, request_preferred_format):
            best_cap = cap

    return best_cap

# public interface to be used by UI layers
def get_audio_devices_public():
    # we support both wasapi and directsound interface, but wasapi has higher precedence
    devices = {}
    for audio_device in device_provider.get_wasapi_devices():
        device = _create_device_public(audio_device, type='wasapi')
        if not device.display_name == 'default':
            devices[device.display_name] = device

    for audio_device in []: # device_provider.get_directsound_devices():
        device = _create_device_public(audio_device, type='directsound')
        if not device.display_name in devices:
            devices[device.display_name] = device
    return sorted(list(devices.values()), key=lambda device: device.display_name.lower())

# public interface to be used by UI layers
def get_video_devices_public():
    devices = []
    include_caps = False
    for video_device in device_provider.get_directshow_camera_devices():
        device = _create_device_public(video_device, type='directshow')
        if include_caps:
            device.caps = []
            unique_caps = {}
            for i in range(video_device.video_capability_count):
                video_cap = video_device.video_capabilities[i]
                unique_by_key = ("%d_%d_%0.2f" % (video_cap.width, video_cap.height, video_cap.framerate))
                unique_caps[unique_by_key] = { 'width': video_cap.width,
                        'height': video_cap.height,
                        'framerate': round(video_cap.framerate, 2) }
            device.caps.extend(sorted(list(unique_caps.values()), key=lambda cap: (cap['width'], cap['height'], cap['framerate']), reverse=True))
        if not device.display_name == 'default':
            devices.append(device)
    return devices


# public interface to be used by UI layers
def get_capturecard_public():
    devices = []
    for video_device in []: # device_provider.get_directshow_capturecard_devices():
        device = _create_device_public(video_device, caps=[])
        unique_caps = {}
        for i in range(video_device.video_capability_count):
            video_cap = video_device.video_capabilities[i]
            unique_by_key = ("%d_%d_%0.2f" % (video_cap.width, video_cap.height, video_cap.framerate))
            unique_caps[unique_by_key] = { 'width': video_cap.width,
                    'height': video_cap.height,
                    'framerate': round(video_cap.framerate, 2) }
        device.caps.extend(sorted(list(unique_caps.values()), key=lambda cap: (cap['width'], cap['height'], cap['framerate']), reverse=True))
        devices.append(device)
    return devices

# private functions
def _get_video_devices_caps(device):
    caps = []
    for i in range(device.video_capability_count):
        video_cap = device.video_capabilities[i]
        cap = {'width': video_cap.width,
               'height': video_cap.height,
               'framerate': round(video_cap.framerate, 2),
               'pixel_format': video_cap.pixel_format.decode()}

        if cap['pixel_format'] == 'MJPEG':
            cap['type'] = 'image/jpeg'
        elif cap['pixel_format'] == 'H264':
            cap['type'] = 'video/x-h264'
        else:
            cap['type'] = 'video/x-raw'

        caps.append(cap)
    return caps


def _get_audio_devices_caps(device):
    caps = []
    for i in range(device.audio_capability_count):
        audio_cap = device.audio_capabilities[i]
        cap = {'channels': audio_cap.channels,
               'sample_rate': audio_cap.sample_rate}
        caps.append(cap)
    return caps


def _create_device_internal(device_provider_device, **kwargs):
    device = Device(display_name=device_provider_device.display_name.decode(),
                    device_id=sha1_hexdigest(device_provider_device.device_id),
                    **kwargs)

    # for internal use, we're going to add device_path,
    # so we can use it for when opening the device
    device.device_path = device_provider_device.device_id.decode()
    return device


def _create_device_public(device_provider_device, **kwargs):
    device = Device(display_name=device_provider_device.display_name.decode(),
                    device_id=sha1_hexdigest(device_provider_device.device_id),
                    **kwargs)

    # We do it here otherwise, we would have to modify each individual source
    # plugin to understand what is device_id: 'default' device.
    # This way is the least effort for now. Mercy takes care of changing device_id: default to real device_id.
    if device.display_name == 'default':
        device.device_id = 'default'
    return device


def _get_audio_devices():
    # we support both wasapi and directsound interface, but wasapi has higher precedence
    devices = {}

    for audio_device in device_provider.get_wasapi_devices():
        device = _create_device_internal(audio_device,
                caps=_get_audio_devices_caps(audio_device),
                type='wasapi')
        devices[device.display_name] = device

    for audio_device in []: # device_provider.get_directsound_devices():
        device = _create_device_internal(audio_device,
                caps=[],
                type='directsound')

        if not device.display_name in devices:
            devices[device.display_name] = device

    return list(devices.values())


def _get_video_devices():
    devices = []
    for video_device in device_provider.get_directshow_devices():
        device = _create_device_internal(video_device,
                caps=_get_video_devices_caps(video_device))
        devices.append(device)
    return devices


def _get_webcam_devices():
    devices = []
    for video_device in []: # device_provider.get_directshow_camera_devices():
        device = _create_device_internal(video_device,
                caps=_get_video_devices_caps(video_device))
        devices.append(device)
    return devices


def _get_capturecard_devices():
    devices = []
    for video_device in []: # device_provider.get_directshow_capturecard_devices():
        device = _create_device_internal(video_device,
                caps=_get_video_devices_caps(video_device))
        devices.append(device)
    return devices


def _find_device(devices, request_device_id):
    if len(devices) == 0:
        return None

    found_device = None
    for device in devices:
        # if we're not requesting for default device, then we're default device,
        # so it doesnt get matched by device_id
        if request_device_id != 'default' and device.display_name == 'default':
            continue
        if request_device_id == device.display_name or request_device_id == device.device_id:
            found_device = device
            break

    # to find the real device that is associated with the default device
    # in this case we only want to match be device_id
    if request_device_id == 'default':
        for device in devices:
            if found_device.device_id == device.device_id and not device.display_name == 'default':
                found_device = device
                break

    return found_device

