import time
from gi.repository import Gst
from .error import PublisherDeviceNotFoundError
from classes import logger as log
from classes.publishers.silent import SilentVideoPublisher
from classes.publishers.browser import BrowserPublisher
from classes.publishers.webcam import WebcamPublisher
from classes.publishers.game import GameCapturePublisher
from classes.publishers.capturecard import CaptureCardPublisher
from classes.publishers.window import WindowCapturePublisher
from classes.publishers.screen import ScreenCapturePublisher
from classes.publishers.media_video import MediaVideoPublisher
from classes.publishers.video_transition import VideoTransitionPublisher
from classes.publishers.audio_device import AudioDevice
from classes.publishers.desktop_audio import DesktopAudio
from classes.stats import qos


class PublisherFactory:

    def __init__(self, ws_send):
        self.ws_send = ws_send
        self._cache = {}
        self._cache_media_id = {}
        self._reserved_media_ids = set()

    def get_publisher(self, widget, current_at, session, target_state=Gst.State.PLAYING, cb=None, on_destroy=None):
        publisher = None

        try:
            started_time = time.time()
            log.debug("get_publisher publisher: %r", widget)
            try:
                widget_id = widget['widget_id']
                cloned_id = widget['cloned_id']
            except KeyError as e:
                log.error("Missing Key %s", e)
                return

            if publisher:
                self.delete_publisher(publisher)

            publisher = self.create_from_payload(widget, session)

            if not publisher:
                log.error("no publisher, cloned_id: %s" % cloned_id)
                return

            publisher.initialize()
            msg = {"url": "/publisher/resolve",
                   "widget_id": widget_id,
                   "cloned_id": cloned_id,
                   "code": 200}
            self.ws_send(msg)
            log.debug("added publisher and successfully running %s in %dms",
                      msg, (time.time() - started_time)*1000)
            if on_destroy:
                publisher.on_destroy = on_destroy
            if cb:
                cb(publisher, widget_id)
            return publisher

        except Exception as e:
            log.exception("failed to get_publisher")
            self.publisher_error(e, publisher)

    def release_publisher(self, publisher):

        if not publisher:
            log.debug("release_publisher: ignoring None publisher")
            return
        try:
            log.debug("release_publisher %r", publisher)
            if publisher.on_destroy:
                publisher.on_destroy(publisher)
                publisher.on_destroy = None
            started_time = time.time()
            self.delete_publisher(publisher)
        except Exception:
            log.error("failed to release_publisher %s", publisher, exc_info=True)

    def debug_log(self, widget_hash, action):
        log.debug("%s Publisher with widget_hash=%s",action, widget_hash)

    def delete_publisher(self, publisher, cb=None):
        try:
            started_time = time.time()
            if publisher.on_destroy:
                publisher.on_destroy(publisher)
                publisher.on_destroy = None
            publisher.destroy()
            log.info("deleted publisher: %s %s in %dms", publisher, publisher.publisher_id, (time.time() - started_time)*1000)
            if cb:
                cb()
        except Exception:
            log.error("failed to delete_publisher %s", publisher, exc_info=True)

    def create_from_payload(self, payload, session):
        widget_type = payload['type']
        if widget_type == 'silent' and is_video_widget(payload):
            return SilentVideoPublisher(payload, session)
        elif widget_type == 'source' and is_video_widget(payload):
            cloned_id = payload['cloned_id']
            log.debug('PublisherFactory create_from_payload type == source w/ cloned_id {}'.format(cloned_id))
            if cloned_id == 'game':
                return GameCapturePublisher(payload, session)
            elif cloned_id == 'webcam':
                return WebcamPublisher(payload, session)
        elif widget_type == 'source' and is_audio_widget(payload):
            cloned_id = payload['cloned_id']
            if cloned_id == 'mic' or cloned_id == 'line_in':
                return AudioDevice(payload, session)
            elif cloned_id == 'desktop-audio':
                return DesktopAudio(payload, session)
        log.error("malformed widget %s", payload)
        return None

    def publisher_error(self, exception, publisher):
        # TODO - this does not belong into the factory - but on the session...
        if not publisher:
            log.error("ignoring publisher error on None publisher")
        if isinstance(exception, PublisherDeviceNotFoundError):
            log.error("%s", exception)
            payload = {"url": "/publisher/error",
                       "widget_id": publisher.widget_id,
                       "message": "device is not connected",
                       "code": 404}
            self.ws_send(payload)
        else:
            log.exception("failed to get_publisher %s %s %s",
                          publisher.cloned_id, publisher.widget_id, type(exception))
            payload = {"url": "/publisher/error",
                       "widget_id": publisher.widget_id,
                       "message": "failed to open the device",
                       "code": 500}
            self.ws_send(payload)

        self.delete_publisher(publisher)


def is_video_widget(widget):
    return widget.get('video', False)


def is_audio_widget(widget):
    return widget.get('audio', False)



