import os

from classes.publishers.visual import VisualPublisher
from classes import logger as log
from classes.branches.fakesink import FakeSink
from classes.branches.cef import Cef

from urllib.parse import urlparse, urlencode, parse_qs, ParseResult


INTERNAL_WIDGET_HOSTS = ['widgets.bebo.com', 'widgets.bebo-ptr.com', 'widgets.bebo-dev.com']


def make_widget_url(widget_type, widget_url, widget_id):
    if not widget_url:
        return ""

    o = urlparse(widget_url)

    is_internal_widget = o.netloc in INTERNAL_WIDGET_HOSTS
    is_localhost = 'localhost' == o.netloc or 'localhost:' in o.netloc # localhost or localhost with ports
    if widget_type == 'external' and not is_internal_widget and not is_localhost:
        return widget_url

    query_params = parse_qs(o.query)
    query_params.update({
        'widget_id': [widget_id],
        'ws_port': [config.PORT],
        'at': [os.getenv('ACCESS_TOKEN')]
        })
    encoded_query_params = urlencode(query_params, doseq=True)
    result = ParseResult(o.scheme, o.netloc, o.path, o.params, encoded_query_params, o.fragment)
    return result.geturl()


class BrowserPublisher(VisualPublisher):

    def __init__(self, publisher_payload, session):
        super(BrowserPublisher, self).__init__(publisher_payload, session)
        self.cef_branch = None
        self.crop_fps = self.session.fps
        self.widget_url = make_widget_url(publisher_payload['type'], publisher_payload['widget_url'], self.widget_id)
        self.dirty_update = False

    def get_values(self, publisher_payload):
        values = super(BrowserPublisher, self).get_values(publisher_payload)
        # values['url'] = publisher_payload['widget_url']
        return values

    def can_update(self, publisher_payload, allow_scale=False):
        new_values = self.get_values(publisher_payload)
        url_different = self.widget_url != make_widget_url(publisher_payload['type'], publisher_payload['widget_url'], self.widget_id)
        if allow_scale:
            dim_different = False
        else:
            dim_different = new_values['load_width'] != self.load_width or new_values['load_height'] != self.load_height
            if self.dirty_update:
                return False
        super_can_update = super(BrowserPublisher, self).can_update(publisher_payload, allow_scale)
        can_update = not dim_different and not url_different and super_can_update
        return can_update

    def update(self, widget, ephermeral=False):

        new_values = self.get_values(widget)
        dim_different = new_values['load_width'] != self.load_width or new_values['load_height'] != self.load_height
        if ephermeral and dim_different:
            self.dirty_update = True
        return super(BrowserPublisher, self).update(widget, ephermeral)

    def can_reload(self):
        return True

    def reload(self):
        self.cef_branch.set_url(self.widget_url)

    def make_src_branches(self):
        self.cef_branch = Cef(self.session.bus, self.session.session_id, self.widget_url, self.load_width, self.load_height)
        return [self.cef_branch]

    def can_pause(self):
        #  don't allow overlay widget updates so we don't play video with audio in the background until we patch cef
        if self.data and self.data.get('type') == "video":
            return False
        return super(BrowserPublisher, self).can_pause()

