import time
import uuid

from gi.repository import Gst

from classes import logger as log
from classes.branches.widget_filter import WidgetFilter
from classes.error import NoBranchError


class Publisher(object):
    """
    public (don't override unless you have a very good reason to):
        initialize():
            Call this first.
        run(wait):
            Put publisher in the gstreamer pipeline to be streamed to twitch.
            wait - don't return until everything in the PLAYING state
        destroy():
            Remove publisher completely, cannot undo. Must make a new publisher.
        update(publisher_payload):
            Update the publisher with a new payload
        can_update(publisher_payload):
            Ask the publisher if a new payload can be updated. Returns True if it can update. False
            if it needs to be removed and readded as a new publisher.
        has_element(element):
            returns True if a specific element is contained in the publisher. False otherwise.

    override:
        (required) make_src_branches():
            Make all branches that are shared in both OFFSCREEN and RUNNING states.
            return array of branches in the order they should be linked. Don't add to pipeline or link them.

        (optional) make_running_branches():
            Make all branches that exist in the RUNNING state and not the OFFSCREEN state.
            first branch must have a link_input_branch() function
            return array of branches in the order they should be linked Don't add to pipeline or linke them.

        (optional) update_running_branches():
           set_values() will have already been called
           define how to handle an update in the RUNNING state 

        (required) link_to_running_sink(branch):
            Links branch into the running sink (usually self.session.video_mixer/audio_mixer)

        (required) get_running_sink_pad():
            return the sink pad that get's linked into. We will wait on this pad for an eos before
            destroying a publisher

        (required) get_running_sink_branch():
            return the sink branch to link into in the RUNNING state. Usually self.session.videomixer or
            self.session.audiomixer. Returned branch must implement cleanup_branch(identifier) which should
            cleanup all of the internal state related to the identifier (widget_id in most cases).

        (optional) get_values():
            return values to set on the publisher. set_values() by default will add all of these values to
            self. Call super to get the values that the superclass handles. For example, VisualPublisher
            calculates all of the cropping data in get_values().

        (optional) set_values():
            gets called in __init__ and in update()
            default behavior will call get_values(). Override this if you need to do any extra work when
            values are set. (i.e. saving values to the windows registry)
    """

    def __init__(self, publisher_payload, session):
        super(Publisher, self).__init__()
        self.publisher_payload = publisher_payload
        self.session = session
        self.publisher_id = str(uuid.uuid4())
        self.set_values(publisher_payload)
        self.on_destroy = None   # called by factory with widget_id
        self._src_branches = []
        self._sink_branches = []
        self._block_probe_id = None
        self._pause_probe_id = None
        self._widget_filters_branches = []
        log.debug("new publisher %s", self)

    def __str__(self):

        data = ['%s id: %s (%s)' % (str(self.__class__)[0:-1], self.publisher_id, self.get_state())]
        if self.cloned_id:
            data.append('(%s)' % self.cloned_id)
        if self.widget_id:
            data.append('widget_id: %s' % self.widget_id)
        if self.widget_hash:
            data.append('widget_hash: %s' % self.widget_hash)
        if self.device_id:
            data.append('device_id: %s' % self.device_id)
        if self.name:
            data.append('name: %s' % self.name)
        return "%s>" % " ".join(data)

    def _add_branches_to_pipeline(self, branches):
        for b in branches:
            try:
                self.session.pipeline.add(b.get_bin())
            except Exception as e:
                if b and hasattr(b, "name"):
                    log.error("error adding branch %s %s",
                              b.name, e, exc_info=True)
                else:
                    log.error("error adding branch %s %s", b, e, exc_info=True)

    def _link_branches(self, branches):

        if len(branches) < 2:
            return

        for index, b in enumerate(branches[1:], 1):
            previous_branch = branches[index - 1]
            b.link_input_branch(previous_branch)

    def _get_ultimate_sink_pad(self):
        return self.get_running_sink_pad()

    def _get_first_sink_pad(self):
        sink_pad = None
        if len(self._sink_branches):
            return self._sink_branches[0].get_sink_pad()
        else:
            sink_pad = self.get_running_sink_pad()
        return sink_pad

    def _is_eos(self, pad):
        """check if the given pad has recieved EOS"""
        return pad.get_last_flow_return() == Gst.FlowReturn.EOS

    def _wait_for_eos(self, pad):
        """wait until this pad has received EOS"""

        ms = 0
        # TODO lower this to 200 ms
        while ms < 500:
            if pad.get_last_flow_return() == Gst.FlowReturn.EOS:
                return
            time.sleep(0.002)
            ms += 2

        log.warning("Wait For EOS timed out %s %s %d",
                    self,
                    pad.get_last_flow_return(),
                    ms)

    def _block_src_and_send_eos(self):
        """blocks src pad and sends eos. returns when eos has reached the last
           sink_pad this is needed to flush data from the pipeline and also to
           prevent crashes
        """
        if not self._src_branches:
            return

        first_src_branch = self._src_branches[0]
        (_, first_state, _) = first_src_branch.get_bin().get_state(2)
        if first_state != Gst.State.PLAYING:
            log.info("src isn't playing, no need to block and send eos %s", self)
            return

        first_src_pad = first_src_branch.get_src_pad()
        src_pad = self._src_branches[-1].get_src_pad()
        first_sink_pad = self._get_first_sink_pad()
        ultimate_sink_pad = self._get_ultimate_sink_pad()

        if self._is_eos(first_src_pad):
            log.info("pad is already eos %s", self)
            self._wait_for_eos(ultimate_sink_pad)
            return

        non_local = {"blocked": False, "received_eos": False}

        def eos_cb(pad, info, user_data):
            event = info.get_event()
            if event is None:
                return Gst.PadProbeReturn.PASS

            if info.get_event().type != Gst.EventType.EOS:
                return Gst.PadProbeReturn.PASS

            non_local["received_eos"] = True
            pad.remove_probe(info.id)
            # tell the mixer to remove the last frame
            return Gst.PadProbeReturn.PASS

        def blocked_cb(pad, info, user_data):

            if non_local["blocked"]:
                return Gst.PadProbeReturn.OK
            non_local["blocked"] = True

            ultimate_sink_pad.add_probe(
                Gst.PadProbeType.EVENT_DOWNSTREAM | Gst.PadProbeType.BLOCK, eos_cb, user_data)
            first_sink_pad.send_event(Gst.Event.new_eos())
            return Gst.PadProbeReturn.OK

        self._block_probe_id = src_pad.add_probe(
            Gst.PadProbeType.IDLE | Gst.PadProbeType.BLOCK_DOWNSTREAM, blocked_cb, None)
        self._wait_for_eos(ultimate_sink_pad)

    def initialize(self):
        """transition NEW --> SRC
        probably don't override
        """
        if self._src_branches or self._sink_branches:
            raise Exception("Can't re-initialize a publisher")

        self._src_branches = self.make_src_branches()
        self._add_branches_to_pipeline(self._src_branches)
        self._link_branches(self._src_branches)

        self._sink_branches = self.make_running_branches()
        self.prepare_running()

        # these are widget.filters
        self._widget_filters_branches = self.make_widget_filters()
        # these filters are sink branches too
        self._sink_branches.extend(self._widget_filters_branches)
        self._sink_branches.extend(self.make_post_filter_branches())
        self._add_branches_to_pipeline(self._sink_branches)
        self._link_branches(self._sink_branches)

        if len(self._sink_branches) and len(self._src_branches):
            last_src_branch = self._src_branches[-1]
            first_running_branch = self._sink_branches[0]
            first_running_branch.link_input_branch(last_src_branch)
            # last_branch = self._sink_branches[-1]
        elif len(self._src_branches):
            pass
            # last_branch = self._src_branches[-1]
        else:
            raise NoBranchError("Failed to link publisher - No sink_branches nor src_branches.")

    def get_max_current_state(self):

        states = []

        for branch in self.get_branches():
            gst_bin = branch.get_bin()
            if not gst_bin:
                continue
            (success, state, pending) = gst_bin.get_state(2)
            if success == Gst.StateChangeReturn.FAILURE:
                states.append(Gst.State.NULL)
                break
            states.append(state)
        if len(states) == 0:
            return Gst.State.NULL
        log.debug("get_max_current_state %r", sorted(states, reverse=True))
        return sorted(states, reverse=True)[0]

    def get_state(self):
        """ if states are pending, return that"""

        states = []

        for branch in self.get_branches():
            gst_bin = branch.get_bin()
            if not gst_bin:
                continue
            (success, state, pending) = gst_bin.get_state(2)
            if success == Gst.StateChangeReturn.FAILURE:
                states.append(Gst.State.NULL)
                break
            if success == Gst.StateChangeReturn.ASYNC:
                states.append(pending)
            else:
                states.append(state)
        if len(states) == 0:
            return Gst.State.NULL
        return sorted(states)[0]

    def play(self):

        branches = self.get_branches(reverse=True)
        last_branch = branches[0]

        if self._pause_probe_id:
            last_sink_pad = last_branch.get_last_sink_pad()
            last_sink_pad.remove_probe(self._pause_probe_id)
            self._pause_probe_id = None

        self.link_to_running_sink(last_branch)
        for branch in branches:
            branch.set_state(Gst.State.PLAYING)

        # FIXME - visual publisher only:
        if self.transition:
            self.animate()

    def pause(self):
        branches = self.get_branches(reverse=False)
        last_branch = branches[-1]
        for branch in branches:
            branch.set_state(Gst.State.PAUSED)

        # some elements pause their task thread if we produce buffers on an
        # un-linked branch, and don't resume it when we go to playing
        def pause_blocked_cb(pad, info, user_data):
            return Gst.PadProbeReturn.DROP

        last_sink_pad = last_branch.get_last_sink_pad()
        if last_sink_pad:
            log.debug("%s blocking last sink pad", self)
            self._pause_probe_id = last_sink_pad.add_probe(Gst.PadProbeType.BLOCK_DOWNSTREAM, pause_blocked_cb, None)
        self.unlink_sink()

    def destroy(self):
        ''' called by the publisher factory - don't call directly please'''
        log.debug("destroying publisher %s", self)

        branches = self.get_branches()

        if self._pause_probe_id and branches:
            last_branch = branches[-1]
            last_sink_pad = last_branch.get_last_sink_pad()
            last_sink_pad.remove_probe(self._pause_probe_id)
            self._pause_probe_id = None

        self._block_src_and_send_eos()

        for b in branches:
            self.session.pipeline.remove(b.get_bin())
            b.destroy()
            b.free_name()

        self._src_branches = []
        self._sink_branches = []
        self.get_running_sink_branch().remove_input_branch(self.publisher_id)
        log.debug("destroying publisher removed branches and set states to NULL")

    def can_update(self, publisher_payload, allow_scale=False):
        if not self.can_update_widget_filters(publisher_payload):
            return False
        return True

    def can_pause(self):
        return True

    def can_reload(self):
        return False

    def reload(self):
        pass

    def can_update_widget_filters(self, publisher_payload):
        # Instead of removing deleted filters or adding new filters, we're just going to
        # remove and readd the entire publisher, so we're not removing and readding the filter element
        # while this publisher branch is PLAYING.
        old_filters = self.publisher_payload.get('filters', [])
        new_filters = publisher_payload.get('filters', [])
        old_filter_ids = [f['stream_widget_filter_id'] for f in old_filters]
        new_filter_ids = [f['stream_widget_filter_id'] for f in new_filters]
        if old_filter_ids != new_filter_ids:
            return False
        for filter_branch in self._widget_filters_branches:
            new_filter = next(
                x for x in new_filters if x['stream_widget_filter_id'] == filter_branch.filter['stream_widget_filter_id'])
            if not filter_branch.can_update(new_filter):
                return False
        return True

    def update(self, publisher_payload, ephermeral=False):
        self.set_values(publisher_payload)
        self.update_running()
        self.update_widget_filters()

    def has_element(self, element):
        bins = [b.get_bin() for b in self.get_branches()]
        # TODO: do we care about elements within bins?
        for b in bins:
            if b == element:
                return True
            for e in b.iterate_elements():
                if e == element:
                    return True

        return False

    def set_values(self, publisher_payload):
        values = self.get_values(publisher_payload)
        for key in values:
            setattr(self, key, values[key])

    def get_values(self, publisher_payload):
        values = {}
        values['publisher_payload'] = publisher_payload
        values['cloned_id'] = publisher_payload['cloned_id']
        values['data'] = publisher_payload.get('data', {})
        values['device_id'] = publisher_payload.get('device_id')
        values['media_id'] = publisher_payload.get('media_id')
        values['name'] = publisher_payload.get('name')
        values['transition'] = publisher_payload.get('transition')
        values['widget_hash'] = publisher_payload['widget_hash']
        values['widget_id'] = publisher_payload['widget_id']
        values['updated_at'] = publisher_payload.get('updated_at')
        return values

    def make_running_branches(self):
        return []

    def make_post_filter_branches(self):
        return []

    def make_src_branches(self):
        raise Exception(
            "make_src_branches needs to be overridden -- don't call super")

    def prepare_running(self):
        pass

    def update_running(self):
        pass

    def update_paused(self):
        pass

    def get_running_sink_pad(self):
        raise Exception(
            "get_running_sink_pad needs to be overridden -- don't call super")

    def get_running_sink_branch(self):
        raise Exception(
            "get_running_sink_branch needs to be overridden -- don't call super")

    def link_to_running_sink(self, branch):
        raise Exception(
            "link_to_running_sink needs to be overridden -- don't call super")

    def unlink_sink(self):
        self.get_running_sink_branch().remove_input_branch(self.publisher_id)

    def get_branches(self, reverse=False):
        branches = []
        branches.extend(self._src_branches)
        branches.extend(self._sink_branches)
        if reverse:
            branches.reverse()
        return branches

    def update_widget_filters(self):
        new_filters = self.publisher_payload.get('filters', [])
        for filter_branch in self._widget_filters_branches:
            new_filter_data = next(
                x for x in new_filters if x['stream_widget_filter_id'] == filter_branch.filter['stream_widget_filter_id'])
            if filter_branch.can_update(new_filter_data):
                filter_branch.update(new_filter_data)

    def make_widget_filters(self):
        branches = []
        filters = self.publisher_payload.get('filters', [])
        if len(filters) > 0:
            # instead of killing the entire branch, we'll just drop the failed filter.
            for filter in filters:
                try:
                    branches.append(WidgetFilter(
                        self.session.bus, self.session.session_id, filter))
                except Exception as e:
                    log.exception('Failed to construct widget filters. %s %s %s',
                                  filter['stream_widget_filter_id'], filter['filter_type'], e)
                    continue
        return branches

