import math

from .publisher import Publisher
from classes import logger as log
from classes.branches.gl_crop import GLCrop
from classes.branches.gl_chroma import GLChroma


class VisualPublisher(Publisher):

    def __init__(self, publisher_payload, session):
        super(VisualPublisher, self).__init__(publisher_payload, session)
        self.crop_fps = 0
        self.crop_branch = None
        self.alpha_branch = None

    def get_values(self, publisher_payload):
        values = super(VisualPublisher, self).get_values(publisher_payload)
        values['position'] = publisher_payload['position']
        values['z_order'] = publisher_payload['z_order']
        values['fps'] = self.session.fps
        pixel_pos = self.calculate_pixel_position(values['position'])
        values['pixel_position'] = pixel_pos['pixel_position']

        values['visible'] = publisher_payload["visible"]
        values['alpha'] = publisher_payload.get("alpha", 1.0)
        values['alpha_dict'] = publisher_payload.get("data", {}).get("alpha", None)
        resolution = publisher_payload.get('resolution', None)
        if resolution:
            split_resolution = resolution.split('x')
            values['load_width'] = int(split_resolution[0])
            values['load_height'] = int(split_resolution[1])
        else:
            values['load_width'] = int(math.floor((publisher_payload['position']['width'] * self.session.width) / 100))
            values['load_height'] = int(math.floor((publisher_payload['position']['height'] * self.session.height) / 100))

        return values

    def get_running_sink_branch(self):
        return self.session.video_mixer

    def get_running_sink_pad(self):
        return self.session.video_mixer.get_sink_pad(self.publisher_id)

    def link_to_running_sink(self, branch):
        alpha = self.alpha
        if not self.visible:
            alpha = 0.0
        log.debug("linking %s to %s alpha %f",
                  branch, self.session.video_mixer.name, alpha)
        self.session.video_mixer.add_input_branch(branch,
                                                  self.publisher_id,
                                                  self.pixel_position,
                                                  self.z_order,
                                                  alpha)

    def can_update(self, publisher_payload, allow_scale=False):
        can_update = super(VisualPublisher, self).can_update(publisher_payload, allow_scale)

        if self.require_crop_branch() and self.crop_branch is None:
            return False

        if not self.require_crop_branch() and self.crop_branch:
            return False

        if self.require_alpha_branch() and self.alpha_branch is None:
            return False

        if not self.require_alpha_branch() and self.alpha_branch:
            return False

        return can_update

    def can_pause(self):

        # can't pause glvideomixer...
        if self.crop_branch is not None and self.crop_branch.has_mixer():
            log.debug("can not pause has mixer %s", self)
            return False
        return True

    def calculate_pixel_position(self, position):
        values = {}
        pixel_position = {}
        canvas_width = self.session.width
        canvas_height = self.session.height
        pixel_position['width'] = int(math.floor((position['width'] * canvas_width) / 100.0))
        pixel_position['height'] = int(math.floor((position['height'] * canvas_height) / 100.0))
        pixel_position['x'] = int(math.floor((position['x'] * canvas_width) / 100.0))
        pixel_position['y'] = int(math.floor((position['y'] * canvas_height) / 100.0))
        values['pixel_position'] = pixel_position
        values['position'] = position
        return values

    def set_property(self, prop, value):
        if prop == "alpha":
            self.alpha = value
            self.update_running()

    def update_running(self):
        super(VisualPublisher, self).update_running()

        alpha = self.alpha
        log.debug("update_running %s alpha: %f visible: %s", self, alpha, self.visible)
        if not self.visible:
            alpha = 0.0

        self.session.video_mixer.update_input_branch(self.publisher_id, self.pixel_position, self.z_order, alpha)

        if self.crop_branch:
            self.crop_branch.update(self.position)

        if self.alpha_branch:
            self.alpha_branch.update(self.alpha_dict)

    def update_paused(self):
        super(VisualPublisher, self).update_paused()

        alpha = self.alpha
        log.debug("update_paused %s alpha: %f visible: %s", self, alpha, self.visible)
        if not self.visible:
            alpha = 0.0

        if self.crop_branch:
            if not self.is_pixel(self.session.width, self.position["width"], self.position["crop_left"]):
                self.position["crop_left"] = 0

            if not self.is_pixel(self.session.width, self.position["width"], self.position["crop_right"]):
                self.position["crop_right"] = 0

            if not self.is_pixel(self.session.height, self.position["height"], self.position["crop_top"]):
                self.position["crop_top"] = 0

            if not self.is_pixel(self.session.height, self.position["height"], self.position["crop_bottom"]):
                self.position["crop_bottom"] = 0
            self.crop_branch.update(self.position)

        if self.alpha_branch:
            self.alpha_branch.update(self.alpha_dict)

    def make_running_branches(self):
        bus = self.session.bus
        session_id = self.session.session_id
        running = []

        if self.require_crop_branch():
            self.crop_branch = GLCrop(bus, session_id, self.crop_fps)
            self.crop_branch.update(self.position)
            running.append(self.crop_branch)

        if self.require_alpha_branch():
            self.alpha_branch = GLChroma(bus, session_id)
            self.alpha_branch.update(self.alpha_dict)
            running.append(self.alpha_branch)

        return running

    def is_pixel(self, resolution, size_percentage, crop_percentage):

        # how many percent is 1 pixel
        relative_percent = 10000/resolution

        # If our crop actually crops a pixel or not
        if resolution * size_percentage * crop_percentage < relative_percent:
            return False
        return True

    def require_crop_branch(self):
        if self.is_pixel(self.session.width, self.position["width"], self.position["crop_left"]):
            return True

        if self.is_pixel(self.session.width, self.position["width"], self.position["crop_right"]):
            return True

        if self.is_pixel(self.session.height, self.position["height"], self.position["crop_top"]):
            return True

        if self.is_pixel(self.session.height, self.position["height"], self.position["crop_bottom"]):
            return True
        
        return False

        # return self.position['crop_top'] or self.position['crop_bottom'] or self.position['crop_right'] or self.position['crop_left']

    def require_alpha_branch(self):
        return self.alpha_dict and self.alpha_dict['enabled']

    def animate(self):
        pass

