import threading
from classes import logger as log
from collections import OrderedDict
from classes.publishers.publisher import Publisher


class SceneUpdate():

    def __init__(self, widgets=None, scene_id=None, current_at=0):

        if not widgets:
            widgets = []
        self.scene_id = scene_id

        for w in widgets:
            w['updated_at'] = current_at

        self.mutex = threading.RLock()  # protects data / widget_id_to_publisher
        self.data = OrderedDict([(w["widget_id"], w) for w in widgets])
        self.widget_id_to_publisher = {}

    def __iter__(self):
        return self.data.values().__iter__()

    def get_publishers(self):
        return list(self.widget_id_to_publisher.values())

    def get_publisher(self, widget):
        widget_id = widget['widget_id']
        p = self.widget_id_to_publisher.get(widget_id, None)
        return p

    def link_publisher(self, widget_id, publisher):
        log.debug("link_publisher %s -> %s", widget_id, publisher)
        self.widget_id_to_publisher[widget_id] = publisher

    def unlink_publisher(self, publisher):
        with self.mutex:
            widget_id = None
            for (w, p) in self.widget_id_to_publisher.items():
                if p == publisher:
                    widget_id = w

            unlinked = None
            if widget_id:
                unlinked = self.widget_id_to_publisher.pop(widget_id, None)

            if unlinked:
                log.debug("unlink_publisher - success - %s", publisher)
            else:
                log.debug("unlink_publisher - not linked - %s", publisher)
            return unlinked

    def append(self, widget):
        widget_id = widget['widget_id']
        self.data[widget_id] = widget

    def remove(self, widget):
        widget_id = widget['widget_id']
        return self.data.pop(widget_id, None)

    def update(self, scene):
        if not scene:
            scene = []
        self.data = OrderedDict([(w["widget_id"], w) for w in scene])
        if scene.scene_id:
            self.scene_id = scene.scene_id

    def get(self, o, d=None):
        with self.mutex:
            if isinstance(o, str):
                return self.data.get(o, d)
            widget_id = o['widget_id']
            return self.data.get(widget_id, d)

    def put(self, widget):
        widget_id = widget['widget_id']
        self.data[widget_id] = widget

    def __contains__(self, o):
        with self.mutex:
            if isinstance(o, str):
                return self.data.__contains__(o)
            widget_id = o['widget_id']
            return self.data.__contains__(widget_id)

    # def get_widget_id_to_publishers(self):
    #     return self.widget_id_to_publisher.items()

    def __repr__(self):

        with self.mutex:
            publishers = set()

            log_line = ["scene_id: %s" % self.scene_id or "unknown"]
            for w in self:
                log_string = "%(cloned_id)s %(widget_id)s" % w
                if w.get("media_id"):
                    log_string += " %(media_id)s" % w
                p = self.get_publisher(w)
                if p:
                    publishers.add(p)
                    log_string += " %s" % p
                log_line.append(log_string)
            for w, p in self.widget_id_to_publisher.items():
                if p not in publishers:
                    log_line.append("bad link %s -> %r" % (w, p))
            return "  ".join(log_line)


class Scene(SceneUpdate):

    pass

