import hashlib
import threading
import time
import json
import re
import os
import sys
import math
PYTHON_DIR = os.path.split(sys.executable)[0]
BASE_DIR = os.path.split(PYTHON_DIR)[0]
BEBO_DLLS_DIR_DEFAULT = os.path.join(BASE_DIR, 'bebodlls')
BEBO_DLLS_DIR = BEBO_DLLS_DIR_DEFAULT

from gi.repository import Gst
from .branches import branch
from .branches.preview import Preview
from .branches.rtmp import Rtmp
from .branches.audio_mixer import AudioMixer
from .branches.wasapi import Wasapi
from .branches.directshow_video import DirectShowVideo
from .branches.gl_video_mixer import GLVideoMixer
from .branches.silent_video import SilentVideo
from .branches.silent_audio import SilentAudio
from .branches.gamecapture import GameCapture
from .branches.cef import Cef
from .error import PublisherDeviceNotFoundError, PublisherFileNotFoundError
from config import options as config
from classes import auth
from classes import device_factory
from classes import logger as log
from classes.stats import qos
from classes.system.gpu import gpu


WEBCAM_PREFER_RESOLUTION = { 'width': 854, 'height': 480, 'framerate': 24 }


def get_default_encoder(nvenc_status):
    encoder = "x264"
    preset = 1
    if nvenc_status == "OKAY":
        encoder = "nvenc"
        preset = None
    return (encoder, preset)


class SessionSetting:

    def __init__(self, mute_desktop_audio=False, prefer_low_quality=False,
                 cam_device_id=None, mic_device_id=None):
        self.mute_desktop_audio = mute_desktop_audio
        self.prefer_low_quality = prefer_low_quality
        self.cam_device_id = cam_device_id
        self.mic_device_id = mic_device_id


class Session:

    def __init__(self, session_id, restart_session, setting):
        self.location = "{}/{}".format(config.RTMP_URL, auth.access_token)
        self.setting = setting
        self.restart_session = restart_session
        self.session_id = session_id
        self.pipeline = None
        self.bus = None
        self.output_retry_timer = None
        self.rtmp_branch = None
        self.async_done = False
        self.publish_queue = []

        (encoder, encoder_preset) = get_default_encoder(gpu.encoder_status)
        self.encoder = encoder
        self.encoder_preset = encoder_preset
        low_quality = self.setting.prefer_low_quality
        if encoder is 'x264':
            low_quality = True
        if low_quality: # TODO(jake): Move output resolution to SessionSetting
            self.width = 852
            self.height = 480
            self.fps = 30
        else:
            self.width = 1280
            self.height = 720
            self.fps = 30
        self.bitrate = 3000
        self.video_mixer = None
        self.audio_mixer = None
        self.audio_sources = {}
        self.video_sources = {}
        log.debug("constructed bebo session")

    def draw_dot(self, name="dot"):
        if self.pipeline is None:
            log.error("No pipeline when asking for dot")
            return
        log.info("draw_dot: %s", name)
        Gst.debug_bin_to_dot_file(self.pipeline, Gst.DebugGraphDetails.ALL, name)

    def start(self):
        self.create_pipeline()
        self.create_mixers()
        self.create_silents()
        self.create_rtmp()
        self.create_initial_src()
        self.pipeline.set_state(Gst.State.PLAYING)
        self.start_remote()

    def stop(self):
        log.info("stopping session")
        self.stop_remote()
        self.pipeline.set_state(Gst.State.NULL)

    def cleanup(self):
        log.info("cleaning up session")
        if self.output_retry_timer:
            self.output_retry_timer.cancel()
        self.destroy_rtmp()
        self.stop()
        self.bus.remove_signal_watch()

    def create_pipeline(self):
        self.pipeline = Gst.Pipeline.new("main-pipeline")

        self.bus = self.pipeline.get_bus()
        self.bus.add_signal_watch()
        self.bus.connect("message", self.on_message)

    def create_mixers(self):
        self.video_mixer = GLVideoMixer(self.bus, self.session_id, self.width, self.height, self.fps, self.encoder != 'x264')
        self.pipeline.add(self.video_mixer.get_bin())
        self.audio_mixer = AudioMixer(self.bus, self.session_id)
        self.pipeline.add(self.audio_mixer.get_bin())

    def create_silents(self):
        width = 1
        height = 1
        pattern = 'black'
        pixel_position = {'x': 0, 'y': 0, 'width': width, 'height': height}
        silent_video = SilentVideo(self.bus, self.session_id, width, height, self.fps, pattern)
        self.pipeline.add(silent_video.get_bin())
        self.video_mixer.add_input_branch(silent_video, "silent_video", pixel_position, 0, 1.0)

        silent_audio = SilentAudio(self.bus, self.session_id, "silence")
        self.pipeline.add(silent_audio.get_bin())
        self.audio_mixer.add_input_branch(silent_audio, "silent_audio")

    def create_initial_src(self):
        cef_width = int(480 / 1280 * self.width)
        cef_height = int(150 / 720 * self.height)
        cef_x = 0
        cef_y = self.height - cef_height
        cef_url = config.WWW_URL + "/overlay?ws_port=" + str(config.PORT)
        cef = Cef(self.bus, self.session_id,
                  cef_url, cef_width, cef_height)
        self.add_video_branch_to_mixer('overlay', cef,
                                       position=self.mk_position(cef_x, cef_y,
                                                                 cef_width, cef_height),
                                       zorder=1)


    def mk_position(self, x, y, width, height):
        return {'x': x, 'y': y, 'width': width, 'height': height}

    def add_audio_branch_to_mixer(self, name, branch, **kwargs):
        if name in self.audio_sources:
            return
        volume = kwargs.get('volume', 1.0)
        self.pipeline.add(branch.get_bin())
        self.audio_mixer.add_input_branch(branch, name, volume)
        self.audio_sources[name] = branch
        (_, pipeline_state, _) = self.pipeline.get_state(2)
        if pipeline_state == Gst.State.PLAYING:
            branch.set_state(Gst.State.PLAYING)

    def add_video_branch_to_mixer(self, name, branch, **kwargs):
        if name in self.video_sources:
            return
        position = kwargs.get('position',
                              self.mk_position(0, 0, self.width, self.height))
        zorder = kwargs.get('zorder', 0)
        alpha = kwargs.get('alpha', 1.0)
        self.pipeline.add(branch.get_bin())
        self.video_mixer.add_input_branch(branch, name, position,
                                          zorder, alpha)
        self.video_sources[name] = branch
        (_, pipeline_state, _) = self.pipeline.get_state(2)
        if pipeline_state == Gst.State.PLAYING:
            branch.set_state(Gst.State.PLAYING)

    def remove_audio_branch_from_mixer(self, name):
        if not name in self.audio_sources:
            return
        self.audio_mixer.remove_input_branch(name)
        ab = self.audio_sources.get(name, None)
        if ab:
            ab.destroy()
            self.pipeline.remove(ab.get_bin())
            self.audio_sources.pop(name)

    def remove_video_branch_from_mixer(self, name):
        if not name in self.video_sources:
            return
        self.video_mixer.remove_input_branch(name)
        vb = self.video_sources.get(name, None)
        if vb:
            vb.destroy()
            self.pipeline.remove(vb.get_bin())
            self.video_sources.pop(name)

    def on_message(self, bus, message):
        try:
            t = message.type
            src = message.src
            name = src.name
            if t == Gst.MessageType.WARNING:
                (error, info) = message.parse_warning()
                error_string = "%s" % error
                info_string = "%s" % info
                log.error("warning: %s, info:%s", error_string, info_string)
            elif t == Gst.MessageType.ERROR:
                return self.on_error(bus, message, src)
            elif t == Gst.MessageType.NEW_CLOCK:
                log.info("NEW_CLOCK: %s" % message.src.get_name())
            elif t == Gst.MessageType.LATENCY:
                pass
            elif t == Gst.MessageType.ASYNC_DONE:
                log.debug("ASYNC_DONE %s" % message.src.get_name())
                log.debug("current thread = %s", threading.current_thread())
                name = message.src.get_name()
                if name == "main-pipeline":
                    log.debug("latency after ASYNC_DONE: %s", self.pipeline.get_latency())
                self.async_done = True
                self.process_publish_queue()
            elif t == Gst.MessageType.STATE_CHANGED:
                pass
            elif Gst.MessageType.STREAM_START or t == Gst.MessageType.STREAM_STATUS:
                pass
            elif t == Gst.MessageType.NEED_CONTEXT or t == Gst.MessageType.HAVE_CONTEXT:
                pass
            elif t == Gst.MessageType.EOS:
                log.warning('Received EOS')
            elif 'audiolevel' not in name:
                log.debug('Unknown message: {}, {}, {}'.format(t, src, str(src)))
        except Exception as e:
            log.error("on_message - Unhandled Exception %s", e, exc_info=True)

    def refresh_session(self):
        log.info('Refreshing Session')

    def restart_session_delayed(self, delay=10.0):
        # TODO: backoff interval
        # TODO: make sure that we should still be streaming
        #       (Today it's fine cause we'd destroy the entire session)
        if self.output_retry_timer:
            self.output_retry_timer.cancel()
        self.output_retry_timer = threading.Timer(delay, self.restart_session,
                                                  [self.session_id, self.setting])
        self.output_retry_timer.start()

    def on_error(self, bus, message, src):
        (error, info) = message.parse_error()
        error_string = "%s" % error
        info_string = "%s" % info

        src_name = str(src)
        if hasattr(src, 'get_name'):
            src_name = src.get_name()

        if self.rtmp_branch and self.rtmp_branch.rtmp_sink == src:
            msg = self.rtmp_branch.get_error_message(src,
                                                     error,
                                                     self.session_id)
            log.info("rtmp error, reconnecting. error: %s info: %s msg: %s",
                     error_string, info_string, msg)
            self.restart_session_delayed(10.0) # It takes awhile to disconnect from remote
            return

        if re.search('d3dgstnvh264enc', src_name, re.IGNORECASE):
            gpu.disable_encoder = True
            log.info('Failed to use hardware encoder (nvenc), disabling the encoder and retrying.')
            self.restart_session_delayed(5.0)
            return

        if re.search('wasapisrc', src_name, re.IGNORECASE):
            self.setting.mute_desktop_audio = True
            log.info('Failed to capture desktop audio, disabling it and retrying.')
            self.restart_session_delayed(5.0)
            return

        self.draw_dot("error_{}".format(int(time.time())))
        log.error("pipeline error. source: %s, error: %s, info:%s", src_name, error_string, info_string)
        self.restart_session_delayed(10.0)

    def create_rtmp(self):
        try:
            self.rtmp_branch = Rtmp(self.bus, self.session_id, self.width, self.height,
                                    self.fps, self.encoder, self.encoder_preset, self.bitrate,
                                    self.location, self.setting)
            self.pipeline.add(self.rtmp_branch.get_bin())
            qos.mark_dirty(self.pipeline)
        except Exception:
            log.exception("failed to create_rtmp")

    def destroy_rtmp(self):
        try:
            if self.rtmp_branch is not None:
                self.pipeline.remove(self.rtmp_branch.get_bin())
                self.rtmp_branch.destroy()
            self.rtmp_branch = None
            qos.mark_dirty(self.pipeline)
        except Exception as e:
            log.error(e, exc_info=True)

    def start_remote(self):
        if self.video_mixer:
            self.video_mixer.add_output_branch("rtmp_video", self.rtmp_branch.get_video_sink_pad())
        if self.audio_mixer:
            self.audio_mixer.add_output_branch("rtmp_audio", self.rtmp_branch.get_audio_sink_pad())
        qos.mark_dirty(self.pipeline)

    def stop_remote(self):
        if self.video_mixer:
            self.video_mixer.remove_output_branch("rtmp_video")
        if self.audio_mixer:
            self.audio_mixer.remove_output_branch("rtmp_audio")

    def get_stats(self):
        return qos.get_stats()

    def process_publish_queue(self):
        while len(self.publish_queue) > 0:
            p = self.publish_queue.pop(0)
            publish_type = p.get("type", "")
            src_type = p.get("src_type", None)
            kwargs = p.get("kwargs", {})
            if publish_type == "publish":
                self.publish_source(src_type, **kwargs)
            else:
                self.unpublish_source(src_type, **kwargs)

    ### Publish / Unpublish sources, can be moved into publisher factory
    def is_publishing(self, source_id):
        if source_id in self.video_sources:
            return True
        if source_id in self.audio_sources:
            return True
        return False

    def is_publishing_type(self, src_type):
        for k, v in self.video_sources.copy().items():
            if v.type == src_type:
                return True
        for k, v in self.audio_sources.copy().items():
            if v.type == src_type:
                return True
        return False

    def publish_source(self, src_type, **kwargs):
        if not self.async_done:
            self.publish_queue.append({ "type": "publish", "src_type": src_type, "kwargs": kwargs})
            return

        # resolution = kwargs.get("resolution", {})
        # position = kwargs.get("position", {})

        if src_type == 'gamecapture':
            window_name = kwargs.get("window_name", None)
            window_class_name = kwargs.get("window_class_name", None)
            self.add_gamecapture_src(window_name, window_class_name)
        elif src_type == 'loopback':
            volume = kwargs.get("volume", 0.7)
            if not self.setting.mute_desktop_audio:
                self.add_audio_src("loopback", volume)
        elif src_type == 'wasapi':
            device_id = kwargs.get("device_id", None)
            volume = kwargs.get("volume", 0.9)
            self.add_audio_src(device_id, volume)
        elif src_type == 'directshow':
            device_id = kwargs.get("device_id", None)
            self.add_webcam_src(device_id)
        else:
            log.info('publish source type not supported')
        # TODO(jake): Return source_id or some sort for identifying the source

    def unpublish_source(self, src_type, **kwargs):
        if not self.async_done:
            self.publish_queue.append({ "type": "unpublish", "src_type": src_type, "kwargs": kwargs})
            return

        source_id = kwargs.get("source_id", None)

        if src_type == "gamecapture":
            if source_id:
                self.remove_video_branch_from_mixer(source_id)
            else:
                for k, v in self.video_sources.copy().items():
                    if v.type == 'gamecapture':
                        self.remove_video_branch_from_mixer(k)
        elif src_type == "loopback":
            self.remove_audio_branch_from_mixer("loopback")
        elif src_type == "wasapi":
            if source_id:
                self.remove_audio_branch_from_mixer(source_id)
            else:
                for k, v in self.audio_sources.copy().items():
                    if k != 'loopback' and v.type == 'wasapi':
                        self.remove_audio_branch_from_mixer(k)
        elif src_type == "directshow":
            if source_id:
                self.remove_video_branch_from_mixer(source_id)
            else:
                for k, v in self.video_sources.copy().items():
                    if v.type == 'direct_show_video':
                        self.remove_video_branch_from_mixer(k)
        else:
            log.info('unpublish source type not supported')

    def add_audio_src(self, device_id, volume):
        if not device_id:
            return
        if device_id in self.audio_sources:
            return
        is_loopback = device_id == "loopback"
        device = device_factory.get_audio_device(device_id)
        if not is_loopback and not device:
            return
        real_device_id = "loopback" if is_loopback else device.device_path
        audio_src = Wasapi(self.bus, self.session_id, real_device_id,
                           True, self.restart_session)
        self.add_audio_branch_to_mixer(device_id, audio_src, volume=volume)

    def add_webcam_src(self, device_id):
        device = device_factory.get_video_device(device_id)
        if not device:
            return

        preferred_caps = device_factory.get_preferred_caps(device,
                                                           WEBCAM_PREFER_RESOLUTION)
        preferred_output_type = preferred_caps.get('type')
        preferred_format = preferred_caps.get('pixel_format')
        preferred_fps = int(preferred_caps.get('framerate'))

        dv = DirectShowVideo(self.bus, self.session_id, device.display_name,
                             device.device_path, preferred_output_type,
                             preferred_caps.get('width'), preferred_caps.get('height'),
                             preferred_fps, preferred_format)

        # TODO(jake): position should be passed in from a layer above
        webcam_width = int(math.floor(0.23 * self.width))
        webcam_height = int(math.floor(webcam_width * 9 / 16))
        webcam_x = int(math.floor(0.005 * self.width))
        webcam_y = int(math.floor(self.height / 2 - webcam_height / 2))
        self.add_video_branch_to_mixer(device_id, dv,
                                       position=self.mk_position(webcam_x, webcam_y,
                                                                 webcam_width, webcam_height),
                                       zorder=2)

    def add_gamecapture_src(self, window_name, window_class_name):
        gamecapture = GameCapture(self.bus, self.session_id,
                                  self.width, self.height, self.fps,
                                  window_name, window_class_name,
                                  True, BEBO_DLLS_DIR)
        name = window_name + ":" + window_class_name
        self.add_video_branch_to_mixer(name, gamecapture,
                                       position=self.mk_position(0, 0, self.width, self.height),
                                       zorder=1)

