import logging
import os
import threading
import uuid
import pycrashpad
import requests
import time
from _version import __version__
from shared import ws
from config import options as config
from classes import auth
from classes import logger as log
from classes.stats.stat_sender import StatSender
from classes.stats.cpu_stats import cpu_stats
from classes.stats.gpu_stats import gpu_stats
from classes.system.gpu import gpu
from classes.scene import Scene, SceneUpdate
from classes.util import TimeoutLock
from classes.session import Session, SessionSetting

logging.getLogger("requests").setLevel(logging.WARNING)
logging.getLogger("urllib3").setLevel(logging.WARNING)



# NOTE: Going to treat SessionCoordinator as the "application" layer
# All bebo-specific-related thing goes in here. don't pollute session.py
# keep that mercy a/v streaming only.
class SessionCoordinator:

    def __init__(self, port):
        self.session_object_map = {}
        self.session_lock = TimeoutLock()
        self.stat_sender = None
        self.ws_manager = ws.WebSocketManager(port, self, self.on_ws_open)
        self.ws_send = self.ws_manager.get_send()
        self.remote_user_setting = {}
        self.local_setting = {}
        gpu_stats.start()
        cpu_stats.start()
        log.info("GpuStats: {}".format(gpu))

    def close(self):
        self.delete_session('match')
        self.ws_manager.close()

    def get_user_access_token(self, message=""):
        self.ws_send({"url": "/user/access_token", "method": "GET", "req_id": str(uuid.uuid4())})

    def get_match(self, match_id):
        self.ws_send({"url": "/match", "method": "GET", "req_id": str(uuid.uuid4()), "match_id": match_id})

    def on_ws_open(self):
        self.ws_send({"url": "/user/me", "method": "GET", "req_id": str(uuid.uuid4())})
        self.get_user_access_token()
        self.ws_send({"url": "/user/settings", "method": "GET", "req_id": str(uuid.uuid4())})

    def on_user_logout(self, message):
        auth.access_token = None # TODO(jake): yeah wtf, we need to fix ws.py to allow multiple handlers
        self.set_crashdump_annotations(message) # TODO(jake): yeah wtf, we need to fix ws.py to allow multiple handlers
        self.get_user_access_token()
        self.remote_user_setting = {}
        self.delete_session('match')

    def on_user_settings(self, message):
        setting = message.get('result', [{}])[0]
        session_id = 'match'
        self.remote_user_setting = setting
        with self.session_lock.acquire_timeout(timeout=20):
            session_obj = self.session_object_map.get(session_id, None)
            if session_obj is not None:
                self.handle_user_settings(session_obj, setting)

    def on_user(self, message):
        payload = message.get('result', [{}])[0]
        self.set_crashdump_annotations(payload) # TODO(jake): yeah wtf, we need to fix ws.py to allow multiple handlers
        active_match_id = payload.get('active_match_id', None)
        platform = payload.get('current_platform', 'pc') or 'pc'
        log.info('sc.on_user active_match_id: {}, platform: {}'.format(active_match_id, platform))
        if active_match_id and platform == 'pc':
            self.add_session('match')
            self.get_match(active_match_id)
        else:
            self.delete_session('match')

    def on_match(self, message):
        payload = message.get('result', [{}])[0]
        session_id = 'match'
        remote_game_id = payload.get("game_id", "fortnite")
        local_game_id = self.local_setting.get("game_id", None)

        # TODO(jake): Ideally we use the a/v pipeline to check for game_id publishing truth
        # instead of local_setting. But mercy is not setup to return the source_id yet.
        if remote_game_id == local_game_id:
            return

        gc_window_name, gc_window_class_name = self.get_gamecapture_info(remote_game_id)
        self.local_setting['game_id'] = remote_game_id

        with self.session_lock.acquire_timeout(timeout=20):
            session_obj = self.session_object_map.get(session_id, None)
            if session_obj:
                # NOTE: Prevent double publishing if add_session held the lock first
                session_obj.unpublish_source("gamecapture")
                session_obj.publish_source("gamecapture",
                                           window_name=gc_window_name,
                                           window_class_name=gc_window_class_name)

    def handle_user_settings(self, session, setting):
        mute_desktop_audio = setting.get("VIPER_mute_desktop_audio", False)
        cam_device_id = setting.get("VIPER_cam_device", None)
        mic_device_id = setting.get("VIPER_mic_device", None)

        if mute_desktop_audio:
            session.unpublish_source('loopback')
        else:
            session.publish_source('loopback')

        if cam_device_id:
            session.unpublish_source('directshow')
            session.publish_source('directshow', device_id=cam_device_id)
        else:
            session.unpublish_source('directshow')

        if mic_device_id:
            session.unpublish_source('wasapi')
            session.publish_source('wasapi', device_id=mic_device_id)
        else:
            session.unpublish_source('wasapi')

    def restart_session(self, session_id, session_setting=None):
        log.info("restarting session %s", session_id)
        self.delete_session(session_id)
        self.add_session(session_id, session_setting)

    def add_session(self, session_id, session_setting=None):

        with self.session_lock.acquire_timeout(timeout=20):
            log.info("adding session_id: %s", session_id)

            if self.session_object_map.get(session_id, None) is not None:
                return None

            if not session_setting:
                session_setting = SessionSetting()

            session = Session(session_id, self.restart_session, session_setting)
            self.session_object_map[session_id] = session
            session.start()

            # Republish sources based on the last setting that we received
            self.handle_user_settings(session, self.remote_user_setting)

            current_game_id = self.local_setting.get('game_id', None)
            if current_game_id:
                gc_window_name, gc_window_class_name = self.get_gamecapture_info(current_game_id)
                session.publish_source("gamecapture",
                                       window_name=gc_window_name,
                                       window_class_name=gc_window_class_name)

            # FIXME: this is wrong, cuz only assume 1 session, which is atm true.
            self.stat_sender = StatSender(session.get_stats, self.send_stats)
            self.stat_sender.start()
            return session

    def delete_session(self, session_id):
        log.info("deleting session %s", session_id)

        with self.session_lock.acquire_timeout(timeout=20):
            session_obj = self.session_object_map.get(session_id, None)
            if session_obj is not None:
                if self.stat_sender:
                    self.stat_sender.stop()
                session_obj.cleanup()
                del self.session_object_map[session_id]

    def get_gamecapture_info(self, game_name):
        gc_window_name = None
        gc_window_class_name = None
        if game_name == "fortnite":
            gc_window_name = "Fortnite  "
            gc_window_class_name = "UnrealWindow"
        elif game_name == "apex":
            gc_window_name = "Apex Legends"
            gc_window_class_name = "Respawn001"
        return (gc_window_name, gc_window_class_name)

    def send_stats(self, stats):
        if not auth.access_token:
            return
        url = "{}{}".format(config.LOG_API_URL, "/mercy/qos")
        body = { 'events': stats }
        headers = {
                'User-Agent': '{}/{}'.format(config.CODE_NAME, __version__),
                'X-Access-Token': auth.access_token
                }
        requests.post(url, json=body, headers=headers)

    def set_crashdump_annotations(self, payload):
        user_id = payload.get('user_id', None)
        username = payload.get('username', None)
        active_match_id = payload.get('active_match_id', None)
        annotations = dict({})
        if user_id:
            annotations['user_id'] = user_id
        if username:
            annotations['username'] = username
        if active_match_id:
            annotations['active_match_id'] = active_match_id
        if len(annotations):
            pycrashpad.set_annotations(annotations)

