import psutil
import os
import re

from . import qos
from classes import logger as log

MIN_REFRESH_CNT = 3
MAX_REFRESH_CNT = 60


class CpuStats:

    def __init__(self):
        self.callbacks = []
        self.procs = {}
        self.proc_cnt = 0
        self.proc_cef_cnt = 0
        self.refresh_cnt = 0
        self.dirty = True
        self.cpu_logical_count = psutil.cpu_count()
        self.cpu_physical_count = psutil.cpu_count(logical=False)
        self.cpu_affinity_count = len(psutil.Process().cpu_affinity())
        # set the cpu interval
        psutil.cpu_percent(interval=None)

    def start(self):
        qos.add_cb(self.get_stats)

    def get_stats(self):
        self.update_process_list()
        proc_stat = dict()
        cpu_stat = dict()
        cpu_stat['logical_cnt'] = self.cpu_logical_count
        cpu_stat['physical_cnt'] = self.cpu_physical_count
        cpu_stat['affinity_cnt'] = self.cpu_affinity_count
        cpu_stat['percent_dec'] = psutil.cpu_percent(interval=None)
        self.get_process_stats(proc_stat)

        # for the app display:
        stats = {"cpu": cpu_stat, "proc": proc_stat}

        # for the app cpu usage display
        stats['bebo_cpu_pct'] = proc_stat['bebo_cpu_dec']
        stats['total_cpu_pct'] = cpu_stat['percent_dec']
        [callback(stats) for callback in self.callbacks]

        return stats

    def get_stats_cb(self):
        qos.add_stats(self.get_stats())

    def add_cb(self, cb):
        self.callbacks.append(cb)

    def remove_cb(self, cb):
        if cb in self.callbacks:
            self.callbacks.remove(cb)

    def add_proc(self, name, proc):
        if not self.procs.get(name):
            self.procs[name] = []
        if proc not in self.procs[name]:
            self.procs[name].append(proc)

    def remove_proc(self, name, proc):
        procs = self.procs.get(name)
        if not procs:
            return
        try:
            procs.remove(proc)
        except ValueError:
            pass

    def add_nw_proc(self, pinfo, proc):
        cmdline = pinfo['cmdline']
        if len(cmdline) == 1:
            name = "browser"
        else:
            name = cmdline[1].replace("", "")
            name = re.sub(r'.js|--type=', "", pinfo['cmdline'][1])
        self.add_proc(name, proc)

    def update_process_list(self):
        self.refresh_cnt += 1
        if self.refresh_cnt < MIN_REFRESH_CNT:
            return
        if not self.dirty and self.refresh_cnt < MAX_REFRESH_CNT:
            return
        self.do_update_process_list()
        self.refresh_cnt = 0
        self.dirty = False
        # import timeit
        # print(timeit.timeit("do_update_process_list()", setup="from __main__ import do_update_process_list", number=10))

    def add_child(self, process):
        children = []
        try:
            children = process.children()
        except (psutil.NoSuchProcess, psutil.ZombieProcess, psutil.AccessDenied):
            pass
        r = [process]
        for c in children:
            r.extend(self.add_child(c))
        return r

    def get_process_list(self):
        current_process = psutil.Process(os.getpid())
        root_process = None
        while current_process:
            root_process = current_process
            try:
                current_process = current_process.parent()
            except (psutil.NoSuchProcess, psutil.ZombieProcess, psutil.AccessDenied):
                break

            if not current_process:
                break
            exe = current_process.exe()
            if not exe:
                break
            if not re.match(r".*(nw\.exe|bebo.exe|bebo_mercy\.exe)$", exe):
                break

        return self.add_child(root_process)

    def do_update_process_list(self):
        self.procs = {}
        self.proc_cnt = 0
        self.proc_cef_cnt = 0

        # this uses about 22% vs using psutil.process_iter() on my machine
        process_iter = self.get_process_list()
        for proc in process_iter:
            self.proc_cnt += 1
            try:
                pinfo = proc.as_dict(attrs=['cmdline', 'exe'])
            except psutil.NoSuchProcess:
                pass
            else:
                exe = pinfo['exe']
                if not exe:
                    continue
                if exe.endswith("bebo_cef.exe"):
                    self.add_proc("cef", proc)
                    self.proc_cef_cnt += 1
                elif exe.endswith("bebo.exe") or exe.endswith("nw.exe"):
                    self.add_nw_proc(pinfo, proc)
                elif exe.endswith("bebo_mercy.exe"):
                    self.add_proc("mercy", proc)

    def get_process_stats(self, proc_stat):
        bebo_cpu = 0
        total_cpu = 0

        for (name, l) in self.procs.items():
            # proc_stat[name] = 0
            category = 0
            for p in l:
                try:
                    # category += p.cpu_percent()
                    pct = p.cpu_percent()
                    category += pct
                except psutil.NoSuchProcess:
                    self.dirty = True
                    self.remove_proc(name, p)

            proc_stat[name + '_cpu_dec'] = category / self.cpu_logical_count
            if name is not "cef":
                bebo_cpu += category
            total_cpu += category

        proc_stat['bebo_cpu_dec'] = bebo_cpu / self.cpu_logical_count
        proc_stat['bebo_total_cpu_dec'] = total_cpu / self.cpu_logical_count
        proc_stat['proc_cnt'] = self.proc_cnt
        proc_stat['proc_cef_cnt'] = self.proc_cef_cnt
        proc_stat['bebo_total_cpu_dec'] = total_cpu / self.cpu_logical_count


cpu_stats = CpuStats()
