import os
import hashlib

from classes import logger as log
from config import options as config
from gi.repository import Gst
from threading import Timer, Thread
import threading
from contextlib import contextmanager
from multiprocessing import TimeoutError


class TimeoutLock(object):
    def __init__(self):
        self._lock = threading.RLock()

    def acquire(self, blocking=True, timeout=-1):
        return self._lock.acquire(blocking, timeout)

    @contextmanager
    def acquire_timeout(self, timeout):

        result = self._lock.acquire(timeout=timeout)
        if result:
            try:
                yield
            finally:
                try:
                    self._lock.release()
                except RuntimeError:
                    pass
            return
        if not timeout:
            timeout = 0.0
        timeout = timeout * 1000
        log.error("timed out after %f ms - clearing lock" % timeout)
        try:
            self._lock.release()
        except RuntimeError:
            pass
        result = self._lock.acquire(blocking=False)
        if result:
            try:
                yield
            finally:
                try:
                    self._lock.release()
                except RuntimeError:
                    pass
                return
        raise TimeoutError("timed out (2) after %f ms - clearing lock" % timeout)

    def release(self):
        self._lock.release()


class SingletonDecorator:

    def __init__(self, klass):
        self.klass = klass
        self.instance = None

    def __call__(self, *args, **kwds):
        if not self.instance:
            self.instance = self.klass(*args, **kwds)
        return self.instance


def link_many(*args):
    can_link = True
    if not len(args):
        return True
    log.debug("link_many linking {} ".format("-->".join(map(lambda x: x.name, args))))
    for index, element in enumerate(args[:-1]):
        next_element = args[index+1]
        can_link = can_link and Gst.Element.link(element, next_element)
        if not can_link:
            log.warning('failed to link {}, {}'.format(element, next_element))

    return can_link


def run_on_thread(func, *args, **kwargs):
    t = Thread(target=func, args=args)
    t.start()
    if kwargs.get('wait', True):
        t.join(kwargs.get('timeout', None))
    return t


# a convenient way to not have to try except everytime
def index_of(list, item, default=-1):
    try:
        return list.index(item)
    except ValueError:
        return default

def sha1_hexdigest(data):
    return hashlib.sha1(data).hexdigest()

def make_bebofs_path():
    # %LOCALAPPDATA%\Bebo-{env}\bebo_fs
    return "{}\\{}\\bebo_fs".format(os.getenv("LOCALAPPDATA"), config.APP_NAME)

def make_chrome_cache_path():
    # %LOCALAPPDATA%\Bebo-{env}\cef
    return "{}\\{}\\cef".format(os.getenv("LOCALAPPDATA"), config.APP_NAME)

def clamp(val, min_, max_):
    return min_ if val < min_ else max_ if val > max_ else val

