import os
import sys
import re

# Versioning
from _version import __version__

# Add python/bin for resolving gi DLLs. Need to do it before import gi
PYTHON_DIR = os.path.split(sys.executable)[0]
PYTHON_BIN_DIR = os.path.join(PYTHON_DIR, 'bin')
BASE_DIR = os.path.split(PYTHON_DIR)[0]
os.environ['PATH'] = os.pathsep.join([PYTHON_BIN_DIR, os.environ['PATH']])

import gi
gi.require_version('Gst', '1.0')

from gi.repository import GLib, Gst, GObject
Gst.init('1.0')


import psutil
import traceback
import pycrashpad
import threading
from threading import Timer, Event
from classes import logger as log
from config import options as config
from classes.session_coordinator import SessionCoordinator


GST_LOG_THREAD_MEMORY = 32000
GST_LOG_THREAD_TIMEOUT_SECONDS = 120

main_loop = None
terminating = False
GST_LOG_LEVEL_EXTRACTOR = re.compile(".*(ERROR|WARNING|WARN|FIXME|INFO|DEBUG|LOG|TRACE).*")
GST_DEBUG_DEFAULT = '*:2,gl*:4,gamecapture:4'
if config.is_local():
    GST_DEBUG = os.getenv("GST_DEBUG", GST_DEBUG_DEFAULT)
else:
    GST_DEBUG = GST_DEBUG_DEFAULT

def initialize_crashpad():
    dump_path = os.path.join(BASE_DIR, 'mercy_crashes')
    log.info('Initializing crashpad: {}'.format(dump_path))
    product = config.CODE_NAME
    if config.ENV != 'prod':
        product += '-' + config.ENV
    annotations = dict({
        'product': product,
        'version': __version__,
    })
    args = []
    rate_limit = config.ENV not in ['local', 'dev']
    if not rate_limit:
        args.append('--no-rate-limit')

    pycrashpad.init_crashpad(dump_path, config.CRASH_DUMP_URL,
                             annotations, args)


def run_main_loop():
    global main_loop
    main_loop = GLib.MainLoop()
    main_loop.run()


gst_log_poller_interval = None
def activate_gst_log_poller():
    logs = Gst.debug_ring_buffer_logger_get_logs()
    Gst.debug_remove_ring_buffer_logger()
    Gst.debug_add_ring_buffer_logger(GST_LOG_THREAD_MEMORY,
                                     GST_LOG_THREAD_TIMEOUT_SECONDS)

    for mln in logs:
        for ln in mln.split('\n'):
            if ln:
                level = 'NONE'
                sre = GST_LOG_LEVEL_EXTRACTOR.search(ln)
                if sre:
                    level = sre.group(1)
                log_func = log.warning
                if level == 'TRACE':
                    log_func = log.debug
                elif level == 'LOG':
                    log_func = log.debug
                elif level == 'DEBUG':
                    log_func = log.debug
                elif level == 'INFO':
                    log_func = log.info
                elif level == 'FIXME':
                    log_func = log.warning
                elif level == 'WARN' or level == 'WARNING':
                    log_func = log.warning
                elif level == 'ERROR':
                    log_func = log.error
                log_func(ln[sre.end(1):].strip())

    if not terminating:
        gst_log_poller_interval = Timer(1.0, activate_gst_log_poller)
        gst_log_poller_interval.start()

def initialize_gstreamer_logger():
    global gst_log_poller_interval
    Gst.debug_set_color_mode(Gst.DebugColorMode.OFF)
    Gst.debug_set_threshold_from_string(GST_DEBUG, True)
    Gst.debug_remove_log_function(None)
    Gst.debug_add_ring_buffer_logger(GST_LOG_THREAD_MEMORY,
                                     GST_LOG_THREAD_TIMEOUT_SECONDS)
    activate_gst_log_poller()

def initialize_gstreamer():
    initialize_gstreamer_logger()

    # If an element fails to load, GStreamer won't ever try to load the element
    # again unless we delete the cache.
    t = Timer(0.0, run_main_loop)
    t.start()
    log.info("Starting Mercy -- Version: %s, Gst Version: %s", __version__, Gst.version_string())

def uncaught_exception_handler(exctype, value, tb):
    log.error("Uncaught Exception %s %s %s", exctype, value, traceback.extract_tb(tb))

if __name__ == '__main__':
    session = None
    try:
        sys.excepthook = uncaught_exception_handler

        initialize_gstreamer()
        initialize_crashpad()

        mercy_process = psutil.Process(os.getpid())
        mercy_process.nice(psutil.HIGH_PRIORITY_CLASS)

        session = SessionCoordinator(config.PORT)

        ev = Event()
        while True:
            ev.wait(1.0)
    except KeyboardInterrupt:
        log.debug('Received keyboard interrupt shutting down process.')
        terminating = True
        if session:
            session.close()
        if main_loop:
            main_loop.quit()
        if gst_log_poller_interval:
            gst_log_poller_interval.cancel()

