import json
import time
import traceback
import uuid
from threading import Thread, Event
from classes import logger as log
from classes import auth
from classes import device_factory
from handlers.log_request import LogRequest
from ws4py.client.threadedclient import WebSocketClient


class WebSocketTask(WebSocketClient):

    def __init__(self, *args, **kwargs):
        super(WebSocketTask, self).__init__(*args, **kwargs)

    def received_message(self, msg):
        try:
            json_payload = json.loads(str(msg))
            MessageTask(json_payload, self.on_message).start()
        except Exception as e:
            log.warning("failed to recv message from worker %s", e)


class MessageTask(Thread):

    def __init__(self, json_payload, func):
        super(MessageTask, self).__init__()
        self.json_payload = json_payload
        self.func = func

    def run(self):
        self.func(self.json_payload)


class WebSocketService:
    route_map = {}

    def __init__(self, port, middleware=None, on_open=None):
        if not middleware:
            middleware = []
        log.info('Websocket service class created port: {}'.format(port))
        self.port = port
        self.task = None
        self.on_open = None
        if on_open:
            self.on_open = on_open

    @staticmethod
    def make_url(port):
        return 'ws://127.0.0.1:%s' % port

    def run_forever(self):
        self.task = WebSocketTask(self.make_url(self.port))
        self.task.on_message = self.on_message
        if self.on_open:
            self.task.opened = self.on_open
        self.task.closed = self.closed
        self.task.connect()
        self.task.run_forever()

    @staticmethod
    def closed(code, reason):
        if type(reason) is bytes:
            reason = reason.decode()
        log.error('ws closed code: %d reason: %s', code, reason)

    def send(self, msg):
        try:
            if hasattr(msg, 'get'):
                url = msg.get('url', msg)
                method = msg.get('method', 'EVENT')
                log.info('%s: %s', method, url)
            else:
                log.info('MSG: %s', msg)

            msg = json.dumps(msg)
            self.task.send(msg)
        except Exception as e:
            log.exception('failed to send ws message')

    def add_route(self, route, fun):
        self.route_map[route] = fun

    def on_message(self, json_payload):
        url = json_payload.get('url', None)
        if not url:
            log.error('no url in json_payload %s' % url)
            return

        start = time.perf_counter()
        status = 404
        if url in self.route_map:
            try:
                result_code = self.route_map[url](json_payload)
                if not result_code:
                    status = 200
            except Exception:
                log.error('failed to call on_message on route_map[url] %s', url, exc_info=True)
                status = 500

            # don't send a response
            # the model is that we are listening to change notifications
            # we can emit changes to existing or new ones

        exec_time = (time.perf_counter() - start) * 1000
        if status != 404:
            log.info('%s %s %.2f ms', url, status, exec_time)


class WebSocketManager:

    def __init__(self, port, session_coordinator, on_open):
        self.port = port
        self.sc = session_coordinator
        self.ws_srv = WebSocketService(port=port, middleware=[LogRequest()],
                                       on_open=on_open)
        self.add_routes()
        self.terminate_event = Event()
        self.runner = Thread(target=ws_run_forever, args=(self.ws_srv, self.terminate_event))
        self.runner.start()

    def add_routes(self):
        log.debug('Adding routes to websocket')
        ws_srv, sc = self.ws_srv, self.sc
        ws_srv.add_route('/user', sc.on_user)
        ws_srv.add_route('/user/me', sc.on_user)
        ws_srv.add_route('/user/signup', sc.get_user_access_token)
        ws_srv.add_route('/user/login', sc.get_user_access_token)
        ws_srv.add_route('/user/logout', sc.on_user_logout)
        ws_srv.add_route('/user/access_token', self.on_user_access_token)
        ws_srv.add_route('/user/settings', sc.on_user_settings)
        ws_srv.add_route('/match', sc.on_match)
        ws_srv.add_route('/device/list', self.on_device_list)

    def on_device_list(self, message):
        data = {}
        req_id = message['req_id']
        video_devices = device_factory.get_video_devices_public()
        audio_devices = device_factory.get_audio_devices_public()
        data['url'] = '/device/list'
        data['ack_id'] = req_id
        data['code'] = 200
        data['video_devices'] = video_devices
        data['audio_devices'] = audio_devices
        self.ws_srv.send(data)

    # FIXME: Need to rethink how to handle auth. Moving forward,
    # This should be in the application layer and not in mercy
    def on_user_access_token(self, message):
        auth.access_token = message['access_token']

    def get_send(self):
        return self.ws_srv.send

    def close(self):
        try:
            self.ws_srv.task.close()
        except Exception as e:
            pass
        self.terminate_event.set()
        self.runner.join()


def ws_run_forever(ws_srv, terminate_event):
    while not terminate_event.is_set():
        try:
            ws_srv.run_forever()
        except Exception as e:
            if isinstance(e, ConnectionRefusedError):
                log.error(e)
            else:
                log.error(traceback.format_exc())
            time.sleep(1)
