import re
from threading import RLock
from gi.repository import Gst
from shared.logger import log

_snake_re = re.compile('([A-Z])')
_names = set()
_name_lock = RLock()
AUDIO_DEV_QUEUE_MAX_SIZE_TIME = 300000000  # 300 ms
AUDIO_MIXER_LATENCY = AUDIO_DEV_QUEUE_MAX_SIZE_TIME - 30000000   # - 30 ms

def to_snake_case(given_value):
    return _snake_re.sub(r'_\1', given_value).lower().replace('__', '_').lstrip('_')


def get_branch_base_name(name):
    return re.sub(r'\d', "", name)


def get_element_base_name(name):
    return re.sub(r'\d', "", name)


def free_name(name):
    with _name_lock:
        _names.discard(name)


def make(element_name, name=None, **kwargs):
    props = {k.replace("_", "-"): v for k, v in kwargs.items()}
    element = Gst.ElementFactory.make(element_name, name)
    if not element:
        raise TypeError('Failed to create element: {}, {} {}'.format(element_name, name, props))
    for k, v in props.items():
        element.set_property(k, v)
    return element


def make_queue(name, queue_name="", **kwargs):
    props = {k.replace("_", "-"): v for k, v in kwargs.items()}
    if queue_name:
        queue_name += "_"
    name = "%s_%squeue" % (name, queue_name)
    queue = Gst.ElementFactory.make("queue", name)
    queue.set_property("max-size-bytes", props.pop("max-size-bytes", 0))
    queue.set_property("max-size-buffers", props.pop("max-size-buffers", 0))
    queue.set_property("max-size-time", props.pop("max-size-time", 0))
    queue.set_property("silent", props.pop("silent", True))
    for k, v in props.items():
        queue.set_property(k, v)
    return queue


def make_default_queue(element_name, name=""):
    return make_queue(element_name, name,
            max_size_buffers=2, leaky=1)


class Element:
    def __init__(self):
        self.name = None
        self.bin = None
        self.is_created = False
        self.elements = []
        with _name_lock:
            self.name = self._mk_name()
            self.bin = Gst.Bin.new(self.name)
        self.id = self.name

    def create_and_link_elements(self):
        """
        The creator of the elements must call this function before using it.
        This can only be called once.
        To give an opportunity to create and link gst elements.
        """
        if self.is_created:
            raise Exception("create_and_link_elements can only be called once!")
        self.is_created = True

    def destroy(self):
        pass

    def make(self, element_name, name=None, **kwargs):
        element = make(element_name, name, **kwargs)
        self.bin.add(element)
        return element

    def make_caps(self, caps_str):
        return self.make("capsfilter", caps=Gst.Caps.from_string(caps_str))

    def make_queue(self, queue_name="", **kwargs):
        queue = make_queue(self.name, queue_name, **kwargs)
        self.bin.add(queue)
        return queue

    def make_default_queue(self, name=""):
        queue = make_default_queue(self.name, name)
        return queue

    def make_default_audio_queue(self, name=""):
        queue = self.make_queue(name, max_size_time=AUDIO_DEV_QUEUE_MAX_SIZE_TIME, leaky=0)
        return queue

    def get_bin_sink_pad(self):
        pad = self.bin.get_static_pad('sink')
        if not pad:
            log.error("sink pad not found %s", self)
            raise Exception("sink pad not found")
        return pad

    def get_bin_src_pad(self):
        pad = self.bin.get_static_pad('src')
        if not pad:
            log.error("src pad not found %s", self)
            raise Exception("src pad not found")
        return pad

    def release_request_pad(self, element, pad):
        structure = Gst.Structure.new_empty("release_request_pad")
        structure.set_value("element", element)
        structure.set_value("pad", pad)
        msg = Gst.Message.new_application(self.bin, structure)
        self.bus.post(msg)

    def link_many(self, *args):
        if not len(args):
            return
        for index, element in enumerate(args[:-1]):
            next_element = args[index+1]
            log.debug("link_many linking %s to %s" %
                      (element.name, next_element.name))
            if not Gst.Element.link(element, next_element):
                log.error('{} failed to link {}, {}'.format(self.name, element, next_element))
                raise error.LinkError('{} failed to link {}, {}'.format(self.name, element, next_element))

    def _mk_name(self, i=0):
        global _names
        if i is 0:
            name = "%s" % to_snake_case(type(self).__name__)
        else:
            name = "%s%d" % (to_snake_case(type(self).__name__), i)
        if name in _names:
            i += 1
            return self._mk_name(i)
        _names.add(name)
        return name

    def get_bin(self):
        return self.bin

    def add_static_ghost_pad(self):
        src_pad = self.elements[-1].get_static_pad('src')
        ghost_src_pad = Gst.GhostPad.new('src', src_pad)
        self.bin.add_pad(ghost_src_pad)

