from gi.repository import Gst
from shared.logger import log
from valkyrie.base import Element
from valkyrie.sources.source import Source
from valkyrie.outputs.output import Output


class MixerElementPad:
    def __init__(self, element, element_pad, mixer_pad):
        self.element = element
        self.element_pad = element_pad
        self.mixer_pad = mixer_pad


class Mixer(Element):
    def __init__(self, mixer_element_name, **kwargs):
        super().__init__()
        self.block_reconfigure_pad = None
        self.block_probe_id = None
        self.mixer = self.make(mixer_element_name, **kwargs)
        self.tee = self.make("tee", allow_not_linked=True)
        self.is_audio = False
        self.is_video = False
        self.source_pads = {}
        self.sink_pads = {}

    def destroy(self):
        super().destroy()
        if self.block_reconfigure_pad and self.block_probe_id:
            self.block_reconfigure_pad.remove_probe(self.block_probe_id)

    def make_filter_elements(self):
        return []

    def add_source(self, source):
        if not isinstance(source, Source):
            raise Exception("Can only add element with type Source")

        if self.is_video and source.is_video:
            sink_pad_id = "sink_video_{}".format(source.id)
            props = source.get_video_mixer_pad_props()
            element_pad_info = self.add_source_to_mixer_pad(sink_pad_id, props)
            source.get_video_src_pad().link(element_pad_info.element_pad)

        if self.is_audio and source.is_audio:
            sink_pad_id = "sink_audio_{}".format(source.id)
            props = source.get_audio_mixer_pad_props()
            element_pad_info = self.add_source_to_mixer_pad(sink_pad_id, props)
            source.get_audio_src_pad().link(element_pad_info.element_pad)

    def remove_source(self, source):
        sid = source.id
        pad_info = self.source_pads.get(sid)
        if not pad_info:
            raise Exception("Source is not in the mixer")

        self.get_bin().remove_pad(pad_info.element_pad)
        self.mixer.release_request_pad(pad_info.mixer_pad)
        del self.source_pads[sid]

    def add_source_to_mixer_pad(self, sink_pad_id, props=None):
        element_sink_pad = Gst.GhostPad.new_no_target(
            sink_pad_id, Gst.PadDirection.SINK)
        element_sink_pad.set_active(True)
        self.bin.add_pad(element_sink_pad)

        mixer_sink_pad = self.mixer.get_request_pad("sink_%u")
        if props:
            for k, v in props.items():
                mixer_sink_pad.set_property(k, v)

        element_sink_pad.set_target(mixer_sink_pad)
        element_pad_info = MixerElementPad(
            source,
            element_sink_pad,
            mixer_sink_pad)

        self.source_pads[source.id] = element_pad_info
        return element_pad_info

    def add_sink(self, sink):
        if not isinstance(sink, Output):
            raise Exception("Can only add element with type Output")

        if self.is_video and sink.is_video:
            src_pad_id = "src_video_{}".format(sink.id)
            mixer_pad_info = self.add_sink_to_mixer_pad(src_pad_id)
            element_sink_pad = sink.get_video_sink_pad()
            mixer_pad_info.element_pad.link(element_sink_pad)

        if self.is_audio and sink.is_audio:
            src_pad_id = "src_audio_{}".format(sink.id)
            mixer_pad_info = self.add_sink_to_mixer_pad(src_pad_id)
            element_sink_pad = sink.get_audio_sink_pad()
            mixer_pad_info.element_pad.link(element_sink_pad)

    def remove_sink(self, sink):
        sid = sink.id
        pad_info = self.sink_pads.get(sid)
        if not pad_info:
            raise Exception("Sink is not in the mixer")
        self.get_bin().remove_pad(pad_info.element_pad)
        self.tee.release_request_pad(pad_info.mixer_pad)
        del self.sink_pads[sid]

    def add_sink_to_mixer_pad(self, src_pad_id, props=None):
        element_src_pad = Gst.GhostPad.new_no_target(
            src_pad_id, Gst.PadDirection.SRC)
        element_src_pad.set_active(True)
        self.bin.add_pad(element_src_pad)

        mixer_src_pad = self.tee.get_request_pad("src_%u")
        element_src_pad.set_target(mixer_src_pad)
        element_pad_info = MixerElementPad(
                sink,
                element_src_pad,
                mixer_src_pad)

        self.sink_pads[source.id] = element_pad_info
        return element_pad_info

    def configure_probe_pad(self):
        before_tee_element = self.elements[-2]
        self.block_reconfigure_pad = before_tee_element.get_static_pad("src")
        self.block_probe_id = self.block_reconfigure_pad.add_probe(
            Gst.PadProbeType.EVENT_UPSTREAM | Gst.PadProbeType.BLOCK,
            self.reconfigure_cb,
            None)

    def reconfigure_cb(self, pad, info, user_data):
        event = info.get_event()
        if event is None:
            return Gst.PadProbeReturn.PASS
        if info.get_event().type != Gst.EventType.RECONFIGURE:
            return Gst.PadProbeReturn.PASS
        return Gst.PadProbeReturn.DROP

