import os
from threading import Timer
from gi.repository import Gst
from shared.logger import log
from .mixers.audio_mixer import AudioMixer
from .mixers.video_mixer import VideoMixer
from .sources.silent_video import SilentVideo
from .sources.silent_audio import SilentAudio
from .sources.gamecapture import GameCapture



class OutputConfig:
    def __init__(self, width, height, fps):
        self.width = width
        self.height = height
        self.fps = fps
        self.master_volume = 1.0
        self.samplerate = 48000
        self.channels = 2
        self.audio_format = "F32LE"
        self.video_format = "RGBA"
        self.video_encoding_preset = "ultrafast"


class Session:
    def __init__(self):
        self.config = OutputConfig(1280, 720, 30)
        self.pipeline = None
        self.bus = None
        self.video_mixers = []
        self.video_sources = []
        self.audio_mixers = []
        self.audio_sources = []
        self.outputs = []

    def draw_dot(self, name="dot"):
        if self.pipeline is None:
            log.error("No pipeline when asking for dot")
            return
        log.info("draw_dot: %s", name)
        Gst.debug_bin_to_dot_file(self.pipeline, Gst.DebugGraphDetails.ALL, name)

    def start(self):
        self.create_pipeline()
        self.create_mixers()
        self.create_silents()
        self.create_outputs()
        self.pipeline.set_state(Gst.State.PLAYING)

        # FIXME this should go to the application layer
        BEBO_DLLS_DIR = os.getenv("BEBO_DLLS")
        self.add_source_to_mixer(self.video_mixers[0], GameCapture(self.config,
            1280, 720,
            window_name="Fortnite  ",
            class_name="UnrealWindow",
            inject_dll_path=BEBO_DLLS_DIR,
            anti_cheat=True))
        Timer(3.0, self.draw_dot, ()).start()

    def stop(self):
        self.pipeline.set_state(Gst.State.PAUSED)
        self.destroy_sources()
        self.destroy_outputs()
        self.destroy_mixers()
        self.destroy_pipeline()

    def create_pipeline(self):
        self.pipeline = Gst.Pipeline.new("main-pipeline")
        self.bus = self.pipeline.get_bus()
        self.bus.add_signal_watch()
        self.bus.connect("message", self.on_message)

    def on_message(self, bus, message):
        t = message.type
        src = message.src
        name = src.name
        if t == Gst.MessageType.WARNING:
            (error, info) = message.parse_warning()
            error_string = "%s" % error
            info_string = "%s" % info
            log.warning("warning: %s, info:%s", error_string, info_string)
        elif t == Gst.MessageType.ERROR:
            (error, info) = message.parse_error()
            error_string = "%s" % error
            info_string = "%s" % info
            log.error("warning: %s, info:%s", error_string, info_string)
        else:
            log.debug('Unknown message: {}, {}, {}'.format(t, src, str(src)))

    def create_mixers(self):
        audio_mixer = AudioMixer(self.config)
        audio_mixer.create_and_link_elements()
        self.pipeline.add(audio_mixer.get_bin())
        self.audio_mixers.append(audio_mixer)

        video_mixer = VideoMixer(self.config)
        video_mixer.create_and_link_elements()
        self.video_mixers.append(video_mixer)
        self.pipeline.add(video_mixer.get_bin())

    def create_silents(self):
        for video_mixer in self.video_mixers:
            self.add_source_to_mixer(video_mixer, SilentVideo(self.config))
        for audio_mixer in self.audio_mixers:
            self.add_source_to_mixer(audio_mixer, SilentAudio(self.config))

    def create_outputs(self):
        pass

    def destroy_pipeline(self):
        self.bus.remove_signal_watch()

    def destroy_sources(self):
        for src in self.audio_sources:
            src.destroy()
        for src in self.video_sources:
            src.destroy()

    def destroy_mixers(self):
        for mixer in self.mixers:
            mixer.destroy()

    def destroy_outputs(self):
        for output in self.outputs:
            output.destroy()

    def warmup_element(self, element):
        # If the pipeline is already playing, then we'll need to set the
        # bin to playing if the pipeline is NOT playing, we'll just add it,
        # and when we set pipeline to playing the bin will automatically
        # set to playing.
        (_, pipeline_state, _) = self.pipeline.get_state(2)
        if pipeline_state == Gst.State.PLAYING:
            element.get_bin().set_state(Gst.State.READY)

    def play_element(self, element):
        # If the pipeline is already playing, then we'll need to set the
        # bin to playing if the pipeline is NOT playing, we'll just add it,
        # and when we set pipeline to playing the bin will automatically
        # set to playing.
        (_, pipeline_state, _) = self.pipeline.get_state(2)
        if pipeline_state == Gst.State.PLAYING:
            element.get_bin().set_state(Gst.State.PLAYING)

    def add_source_to_mixer(self, mixer, element):
        element.create_and_link_elements()
        if element.is_audio:
            self.audio_sources.append(element)
        if element.is_video:
            self.video_sources.append(element)
        self.pipeline.add(element.get_bin())
        mixer.add_source(element)
        self.play_element(element)

