import json
import logging
import re
import hmac
import hashlib

from .config import SLACK_SIGNING_SECRET, slack_client, lambda_client, MONITOR_ENDPOINT
from .models import NotificationModel
from .utils import get_user_info_by_login

from .ask_working import on_threaded_message

CHANNEL_REGEX = re.compile(r'twitch\.tv\/([a-zA-Z0-9_]+)')

def verify_signature(event):
    timestamp = event['headers']['X-Slack-Request-Timestamp']
    signature = event['headers']['X-Slack-Signature']
    data = event['body']
    # Verify the request signature of the request sent from Slack
    # Generate a new hash using the app's signing secret and request data

    # Compare the generated hash and incoming request signature
    req = str.encode('v0:' + str(timestamp) + ':') + str.encode(data)
    request_hash = 'v0=' + hmac.new(
        str.encode(SLACK_SIGNING_SECRET),
        req, hashlib.sha256
    ).hexdigest()
    return hmac.compare_digest(request_hash, signature)


def make_response(data={}):
    return {
        'statusCode': 200,
        'headers': {'Content-Type': 'application/json'},
        'body': json.dumps(data)
    }

def handle_slack_event(event):
    if not verify_signature(event):
        raise Exception('invalid signature')

    logging.info("header: %s", event['headers'])
    if 'X-Slack-Retry-Num' in event['headers'] and event['headers']['X-Slack-Retry-Num']:
        logging.warning("ignoreing retry message: %s", event)
        return make_response()

    data = json.loads(event['body'])
    logging.info("received: %s", data)
    if data['type'] == 'url_verification':
        return make_response({"challenge": data['challenge']})

    elif data['type'] == 'event_callback':
        if data['event']['type'] == 'message' and data['event'].get('thread_ts'):
            on_threaded_message(data['event'])
        elif data['event']['type'] == 'app_mention':
            handle_app_mention(data)
    return make_response()

def handle_app_mention(data):
    #  @mgstbot ...
    print("handle_app_mention w/ data: %s", data)
    msg_ts = data['event']['ts']
    slack_channel = data['event']['channel']

    responses = []

    split = data['event']['text'].split()

    groups = {e for e in CHANNEL_REGEX.findall(data['event']['text'])}
    if groups:
        if 'delete' in split:
            responses.extend(delete_channels(groups, slack_channel, msg_ts))
        else:
            responses.extend(add_channels(groups, slack_channel, msg_ts))
    elif 'list' in split:
        responses.append(list_channels(slack_channel, msg_ts))
    elif 'thx' in split or 'thanks' in split or 'thank' in split:
        responses.append(slack_client.chat_postMessage(
            channel=slack_channel,
            thread_ts=msg_ts,
            text=':welcome: anytime pardner'
        ))
    else:
        responses.append(slack_client.chat_postMessage(
            channel=slack_channel,
            thread_ts=msg_ts,
            text=':vohiyo1: ohai'
        ))

    for response in responses:
        if not response['ok']:
            logging.warning('failed to post response %s', response)


def add_channels(channels, slack_channel, msg_ts):
    responses = []
    for channel in channels:
        user_data = get_user_info_by_login(channel)['data']
        print('add_channel adding %s -- user_data: %s' % (channel, user_data))
        user = user_data[0]

        user_id = user['id']

        noti = NotificationModel(user_id, slack_channel, user_login=channel)
        noti.save()

        responses.append(slack_client.chat_postMessage(
            channel=slack_channel,
            thread_ts=msg_ts,
            text=':calling: tracking %s' % channel)
        )

    return responses

def delete_channels(channels, slack_channel, msg_ts):
    responses = []
    for channel in channels:
        deleted = 0
        user = get_user_info_by_login(channel)['data'][0]

        user_id = user['id']

        for noti in NotificationModel.scan(NotificationModel.user_id == user_id):
            if noti.slack_ts and noti.slack_channel_id:
                try:
                    slack_client.chat_delete(
                        channel=noti.slack_channel_id,
                        ts=noti.slack_ts
                    )
                except Exception as e:
                    logging.warning(
                        "failed to delete message %s %s", noti.user_login, e)
            noti.delete()
            deleted += 1
        responses.append(slack_client.chat_postMessage(
            channel=slack_channel,
            thread_ts=msg_ts,
            text=':no_mobile_phones: deleted %d tracker for %s' % (deleted, channel))
        )

    lambda_client.invoke(
        FunctionName=MONITOR_ENDPOINT,
        InvocationType='Event'
    )
    return responses

def list_channels(slack_channel, msg_ts):
    users = list(
        set(map(lambda noti: noti.user_login, NotificationModel.scan())))

    return slack_client.chat_postMessage(
        channel=slack_channel,
        thread_ts=msg_ts,
        text='I am currently tracking: %s' % users
    )
