import os
import logging
import collections
import requests
from requests.utils import requote_uri
from mgst.models import UserModel, NotificationModel, StreamsModel
from mgst.config import slack_client, SUMMARY_CHAN
from mgst.utils import spade_send, get_stream_by_user_id, parse_twitch_timestamp, how_long_since, get_user_info_by_id, get_game_by_id

SPADE_BLUEPRINT = 'mgst_bot_streamer_tracker'
MIN_VIEWER = 2000
UPDATE_INTERVAL = 15  # minutes
MOBILE_GROWTH_CHAN = 'GM4R19VRS'

def send_notification(user, noti, broadcast):
    game = get_game_by_id(broadcast['game_id'])['name']
    twitch_link = f"https://www.twitch.tv/{broadcast['user_name']}"
    attachments = [
        {
            "author_name": f"{broadcast['user_name']} - Twitch",
            "author_link": twitch_link,
            "author_icon": "https://www.twitch.tv/favicon.ico",
            "text": f"<{twitch_link}|*{broadcast['user_name']}*> is streaming <{requote_uri('https://www.twitch.tv/directory/game/' + game)}|*{game}*> with *{broadcast['viewer_count']}* viewers :pogchamp:",
            "thumb_url": broadcast['thumbnail_url'].format(height=320, width=569),
            "color": "#6441A5"
        }
    ]

    try:
        if user.state != 'live' or not noti.slack_ts or not noti.slack_channel_id:
            logging.info(
                f'sending notification for {broadcast["user_name"]} to {noti.slack_channel}')

            response = slack_client.chat_postMessage(
                channel=MOBILE_GROWTH_CHAN,
                attachments=attachments)
        else:
            logging.info(
                f'updating notification for {broadcast["user_name"]} to {noti.slack_channel}')
            response = slack_client.chat_update(
                channel=noti.slack_channel_id,
                ts=noti.slack_ts,
                attachments=attachments)

        if not response["ok"]:
            logging.error(response)

        noti.update(actions=[
            NotificationModel.slack_ts.set(response['ts']),
            NotificationModel.slack_channel_id.set(
                response['channel'])
        ])
    except Exception:
        logging.exception(
            "Failed to send notification fro %s to %s", user, noti)


def stream_monitor():
    logging.info("update starting")

    user_notis = collections.defaultdict(list)
    for noti in NotificationModel.scan():
        user_notis[noti.user_id].append(noti)

    user_succeeded = 0
    for user_id, notis in user_notis.items():
        try:
            user = UserModel.get(user_id)
        except UserModel.DoesNotExist:
            user = UserModel(user_id, state='offline')
            user.save()

        try:
            res = get_stream_by_user_id(user_id)

            if res['data'] and res['data'][0]['type'] == 'live':
                broadcast = res['data'][0]
                stream_id = broadcast['id']
                started_at = parse_twitch_timestamp(broadcast['started_at'])

                try:
                    stream = StreamsModel.get(stream_id)
                except StreamsModel.DoesNotExist:
                    stream = StreamsModel(stream_id, user_id=user_id)

                started_ago = how_long_since(started_at)
                ccv = max(stream.ccv, broadcast['viewer_count'])

                stream_actions = [
                    StreamsModel.minutes_live.set(started_ago),
                    StreamsModel.ccv.set(ccv),
                    StreamsModel.hours_watched.set(
                        stream.hours_watched + (broadcast['viewer_count'] * UPDATE_INTERVAL / 60))
                ]

                if broadcast['game_id']:
                    # for some reason we see a game_id of 0 sometimes :/
                    stream_actions.append(
                        StreamsModel.game_id.set(broadcast['game_id']))

                stream.update(actions=stream_actions)
                stream.save()

                if broadcast['viewer_count'] > MIN_VIEWER:
                    [send_notification(user, noti, broadcast)
                     for noti in notis]

                user.update(actions=[
                    UserModel.state.set('live'),
                ])
            else:
                user.update(actions=[
                    UserModel.state.set('offline'),
                ])
            user_succeeded += 1
        except Exception:
            logging.exception("Failed to process user %s", user)

    logging.info(
        f"update finished, processed {user_succeeded}/{len(user_notis)} users")


def dump_stream_tracker():
    notifs = list(NotificationModel.scan())
    events = []
    for noti in notifs:
        try:
            username = get_user_info_by_id(noti.user_id)['data'][0]['login']
            events.append({
                'event': SPADE_BLUEPRINT,
                'properties': {
                    'user_login': username,
                    'channel_id': int(noti.user_id)
                }
            })
        except Exception as e:
            logging.exception("Failed to add %s to spade list -- %s" % (noti, e))

    logging.info("streamer list: %s", events)
    if os.getenv('MGST_ENV') == 'production' and events:
        spade_send(events)
