from functools import reduce
import logging
import collections
import time

from mgst.models import StreamsModel, UserModel
from mgst.utils import get_utc_now, get_user_info_by_id, pretty_time, get_game_by_id, pretty_number
from mgst.config import SUMMARY_CHAN, slack_client


def send_summary():
    streams = list(StreamsModel.scan(StreamsModel.slack_ts.does_not_exist()))
    if not streams:
        print("no new unsent streams -- exiting")
        return

    header, long_msg = generate_text(streams)
    if not long_msg or not header:
        print("new unsent streams but no generated message..?")
        return

    print('Sending "\n%s\n%s\n" to %s' % (header, long_msg, SUMMARY_CHAN))

    resp = slack_client.chat_postMessage(
        channel=SUMMARY_CHAN,
        text=header,
        attachments=[{
            "text": long_msg
        }]
    )

    if resp['ok']:
        for stream in streams:
            stream.update(actions=[StreamsModel.slack_ts.set(resp['ts'])])
            time.sleep(0.1)


def generate_text(streams):
    streams_by_id = collections.defaultdict(list)
    for s in streams:
        streams_by_id[s.user_id].append(s)

    data = []
    total_hw = int(sum(map(lambda s: s.hours_watched, streams)))
    for user_id, own_streams in streams_by_id.items():
        try:
            username = get_user_info_by_id(user_id)['data'][0]['login']
            sum_minutes_live = sum([stream.minutes_live for stream in own_streams])
            sum_minutes_watched = sum([stream.hours_watched * 60 for stream in own_streams])

            accv = sum_minutes_watched / sum_minutes_live

            ccv_str = '*%s CCV*' % pretty_number(int(accv))

            data.append({
                'username_link': '<%s|*%s*>' % ('https://twitch.tv/%s' % username, username),
                'username': username,
                'length': pretty_time(sum_minutes_live),
                'stream_count': f"{len(own_streams)} stream{'s' if len(own_streams) > 1 else ''}",
                'ccv_str': ccv_str,
                'ccv': accv
            })

        except Exception:
            logging.exception('Failed to add %s to summary' % (user_id))

    entries = sorted(data, key=lambda e: - e['ccv'])

    lines = []
    for e in entries:
        row = f"{e['username_link']} | " + \
            f"{e['ccv_str']} | " + \
            f"{e['length']} | " + \
            f"{e['stream_count']}"

        lines.append(row)
    msg = "\n".join(lines)

    header = f"Yesterday Summary: *{pretty_number(total_hw)}* Hours Watched. Top Streamers:"

    return header, msg
