import datetime
import logging
import json
import base64
from functools import lru_cache
from time import sleep
import requests
from mgst.config import TWITCH_CLIENT_ID, TWITCH_CLIENT_SECRET
from cachetools import cached, TTLCache

date_format = '%Y-%m-%dT%H:%M:%SZ'
MAX_TRIES = 5
LRU_MAXSIZE = 1000

def get_date():
    return str(datetime.datetime.utcnow().date())


def get_sheet_name():
    rn = datetime.datetime.utcnow()
    return str((rn - datetime.timedelta(days=rn.weekday())).date())


def get_weekday_offset():
    return datetime.datetime.utcnow().weekday()


def get_utc_now():
    return datetime.datetime.utcnow()


def spade_send(events):
    try:
        event_json = json.dumps(events, separators=(',', ':'))
        encoded_event = base64.b64encode(bytes(event_json, 'utf-8'))
        res = requests.post('https://spade.twitch.tv',
                            data={'data': encoded_event})
        if res.status_code != 204:
            logging.error(
                'Spade request failed[%d] %s', res.status_code, res.text)
    except Exception as e:
        logging.exception('Spade send failed')

# cache for 5 minutes
@cached(cache=TTLCache(maxsize=LRU_MAXSIZE, ttl=300))
def get_request_headers():
    querystring = {
        "client_id": TWITCH_CLIENT_ID,
        "client_secret": TWITCH_CLIENT_SECRET,
        "grant_type": "client_credentials"
    }

    response = requests.request("POST", "https://id.twitch.tv/oauth2/token", params=querystring)
    response.raise_for_status()

    logging.info("OAuth Resp: %s", response.json())

    return {
        'Authorization': f"Bearer {response.json()['access_token']}",
        'Client-ID': TWITCH_CLIENT_ID
    }

@lru_cache(maxsize=LRU_MAXSIZE)
def get_user_info_by_login(login):
    res = requests.get('https://api.twitch.tv/helix/users',
                       params={'login': login},
                       headers=get_request_headers()
                       )
    res.raise_for_status()
    return res.json()

@lru_cache(maxsize=LRU_MAXSIZE)
def get_user_info_by_id(user_id, t=0):
    try:
        resp = requests.get('https://api.twitch.tv/helix/users',
                            params={'id': user_id},
                            headers=get_request_headers()
                            )
        resp.raise_for_status()
        return resp.json()
    except requests.exceptions.HTTPError as e:
        if resp.status_code == 429 and t < MAX_TRIES:
            logging.warn('Retrying get_user_info_by_id user_id: %s #%s' % (user_id, t))
            sleep(1)
            return get_user_info_by_id(user_id, t+1)

        raise e


def get_stream_by_user_id(user_id):
    res = requests.get('https://api.twitch.tv/helix/streams',
                       params={'user_id': user_id},
                       headers=get_request_headers()
                       )
    res.raise_for_status()
    return res.json()


def how_long_since(dttm):
    delta = datetime.datetime.utcnow() - dttm
    return delta.seconds // 60


def parse_twitch_timestamp(s):
    return datetime.datetime.strptime(s, date_format)

def pretty_number(n):
    return f"{n:,}"


def pretty_time(minutes):
    seconds = minutes * 60
    hours, remainder = divmod(seconds, 3600)
    minutes, seconds = divmod(remainder, 60)

    res = ""

    if hours > 0:
        res = "%sh" % hours

    if minutes > 0:
        res += "%sm" % minutes

    return res

@lru_cache(maxsize=LRU_MAXSIZE)
def get_game_by_id(game_id, t=0):
    try:
        resp = requests.get(
            'https://api.twitch.tv/helix/games',
            params={'id': game_id},
            headers=get_request_headers()
        )

        resp.raise_for_status()

        return resp.json()['data'][0]
    except requests.exceptions.HTTPError as e:
        if resp.status_code == 429 and t < MAX_TRIES:
            logging.warn('Retrying get_game_by_id game_id: %s #%s' % (game_id, t))
            sleep(1)
            return get_game_by_id(game_id, t+1)

        raise e
